/**
 * ts-agent-sdk Public Holidays API
 *
 * Wrapper for Nager.Date Public Holiday API.
 * @see https://date.nager.at/Api
 *
 * No API key required - this is a free public API.
 */

import { get } from '../base';

const NAGER_API_BASE = 'https://date.nager.at/api/v3';

export interface PublicHoliday {
  date: string;
  localName: string;
  name: string;
  countryCode: string;
  fixed: boolean;
  global: boolean;
  counties: string[] | null;
  launchYear: number | null;
  types: ('Public' | 'Bank' | 'School' | 'Authorities' | 'Optional' | 'Observance')[];
}

export interface CountryInfo {
  commonName: string;
  officialName: string;
  countryCode: string;
  region: string;
  borders: string[] | null;
}

/**
 * Get public holidays for a specific year and country.
 *
 * @param year - The year (e.g., 2025)
 * @param countryCode - ISO 3166-1 alpha-2 country code (e.g., 'AU', 'US', 'GB')
 *
 * @example
 * const holidays = await getPublicHolidays(2025, 'AU');
 */
export async function getPublicHolidays(year: number, countryCode: string): Promise<PublicHoliday[]> {
  const url = `${NAGER_API_BASE}/PublicHolidays/${year}/${countryCode.toUpperCase()}`;
  return get<PublicHoliday[]>(url);
}

/**
 * Get the next public holiday for a country.
 *
 * @param countryCode - ISO 3166-1 alpha-2 country code
 *
 * @example
 * const next = await getNextPublicHoliday('AU');
 * if (next) console.log(`Next holiday: ${next.name} on ${next.date}`);
 */
export async function getNextPublicHoliday(countryCode: string): Promise<PublicHoliday | null> {
  const url = `${NAGER_API_BASE}/NextPublicHolidays/${countryCode.toUpperCase()}`;

  try {
    const holidays = await get<PublicHoliday[]>(url);
    return holidays[0] || null;
  } catch {
    return null;
  }
}

/**
 * Get the next public holidays (up to a limit) for a country.
 *
 * @param countryCode - ISO 3166-1 alpha-2 country code
 * @param limit - Maximum number of holidays to return (default: 5)
 */
export async function getNextPublicHolidays(
  countryCode: string,
  limit: number = 5
): Promise<PublicHoliday[]> {
  const url = `${NAGER_API_BASE}/NextPublicHolidays/${countryCode.toUpperCase()}`;

  try {
    const holidays = await get<PublicHoliday[]>(url);
    return holidays.slice(0, limit);
  } catch {
    return [];
  }
}

/**
 * Check if a specific date is a public holiday.
 *
 * @param date - The date to check (YYYY-MM-DD format or Date object)
 * @param countryCode - ISO 3166-1 alpha-2 country code
 */
export async function isPublicHoliday(
  date: string | Date,
  countryCode: string
): Promise<{ isHoliday: boolean; holiday?: PublicHoliday }> {
  const dateStr = date instanceof Date ? date.toISOString().split('T')[0] : date;
  const year = parseInt(dateStr.split('-')[0], 10);

  const holidays = await getPublicHolidays(year, countryCode);
  const holiday = holidays.find((h) => h.date === dateStr);

  return {
    isHoliday: !!holiday,
    holiday,
  };
}

/**
 * Get list of available countries.
 */
export async function getAvailableCountries(): Promise<CountryInfo[]> {
  const url = `${NAGER_API_BASE}/AvailableCountries`;
  return get<CountryInfo[]>(url);
}

/**
 * Get country information.
 *
 * @param countryCode - ISO 3166-1 alpha-2 country code
 */
export async function getCountryInfo(countryCode: string): Promise<CountryInfo> {
  const url = `${NAGER_API_BASE}/CountryInfo/${countryCode.toUpperCase()}`;
  return get<CountryInfo>(url);
}

/**
 * Check if today is a public holiday.
 *
 * @param countryCode - ISO 3166-1 alpha-2 country code
 */
export async function isTodayPublicHoliday(
  countryCode: string
): Promise<{ isHoliday: boolean; holiday?: PublicHoliday }> {
  const today = new Date().toISOString().split('T')[0];
  return isPublicHoliday(today, countryCode);
}

// Convenience export for common country codes
export const COUNTRY_CODES = {
  AUSTRALIA: 'AU',
  AUSTRIA: 'AT',
  BELGIUM: 'BE',
  BRAZIL: 'BR',
  CANADA: 'CA',
  FRANCE: 'FR',
  GERMANY: 'DE',
  IRELAND: 'IE',
  ITALY: 'IT',
  JAPAN: 'JP',
  NETHERLANDS: 'NL',
  NEW_ZEALAND: 'NZ',
  SPAIN: 'ES',
  SWITZERLAND: 'CH',
  UNITED_KINGDOM: 'GB',
  UNITED_STATES: 'US',
} as const;
