/**
 * ts-agent-sdk Configuration
 *
 * Environment variables:
 * - SDK_MODE: 'local' | 'remote' | 'auto' (default: 'auto')
 * - SDK_BASE_URL: Target Worker URL (required for remote mode)
 * - SDK_API_TOKEN: Bearer token for authentication
 */

export type ExecutionMode = 'local' | 'remote' | 'auto';

export interface SDKConfig {
  mode: ExecutionMode;
  baseUrl: string;
  apiToken: string;
}

/**
 * Load SDK configuration from environment variables.
 * Call this at startup or let the client handle it automatically.
 */
export function loadConfig(): SDKConfig {
  const mode = (process.env.SDK_MODE as ExecutionMode) || 'auto';
  let baseUrl = process.env.SDK_BASE_URL || '';

  // Auto-detect: if no URL provided or explicitly local, use localhost
  if (mode === 'auto' && !baseUrl) {
    baseUrl = 'http://localhost:8787';
  } else if (mode === 'local') {
    baseUrl = baseUrl || 'http://localhost:8787';
  }

  return {
    mode,
    baseUrl,
    apiToken: process.env.SDK_API_TOKEN || '',
  };
}

/**
 * Validate configuration and throw helpful errors if misconfigured.
 */
export function validateConfig(config: SDKConfig): void {
  if (config.mode === 'remote' && !config.baseUrl) {
    throw new Error(
      'SDK_BASE_URL is required when SDK_MODE=remote. ' +
        'Set it to your deployed Worker URL (e.g., https://app.workers.dev)'
    );
  }

  if (config.mode === 'remote' && !config.apiToken) {
    throw new Error(
      'SDK_API_TOKEN is required when SDK_MODE=remote. ' +
        'Generate an API token from your app and set it in the environment.'
    );
  }
}
