# Tutorial 4: Magic System

Learn about resource management and modular data architecture by adding a complete spellcasting system to D&D combat!

## Overview

This tutorial teaches you how to implement resource management systems and design modular, data-driven architectures. You'll extend the combat system from Tutorial 3 with a complete spellcasting system featuring spell attacks, saving throws, spell slot tracking, and multiple damage types—all driven by external JSON data files.

## What You'll Learn (Skill Building)

- **Resource management**: Tracking consumable resources (spell slots) that deplete and refresh
- **Modular data architecture**: Separating reference data (JSON files) from application logic
- **Hybrid storage patterns**: Using SQLite + JSON for flexible, queryable data
- **Conditional mechanics**: Different resolution paths (spell attacks vs saving throws vs auto-hit)
- **Data seeding workflows**: Loading external data sources into databases
- **Type systems**: Handling multiple damage types and resolution mechanics
- **Database schema evolution**: Adding JSON columns for flexible data structures

## What You'll Build (D&D Feature)

A complete spellcasting system that:
- Manages a spell database with 10 spells (5 cantrips + 5 leveled spells)
- Loads spell data from JSON files into SQLite for efficient querying
- Tracks spell slots per character (JSON column in database)
- Implements three spell resolution mechanics:
  - **Spell attacks**: Roll d20 + spell attack bonus vs AC (Fire Bolt, Ray of Frost)
  - **Saving throws**: Target rolls vs spell save DC (Burning Hands, Thunderwave)
  - **Auto-hit**: No roll needed (Magic Missile)
- Manages spell slot consumption for leveled spells (cantrips are free)
- Restores spell slots on long rest
- Integrates seamlessly with Tutorial 3's combat system
- Calculates spellcasting stats based on character class (INT for wizards, WIS for clerics, etc.)

## Prerequisites

- Claude Code installed and configured
- Completed Tutorial 1 (dice rolling), Tutorial 2 (character stats), and Tutorial 3 (combat)
- Basic familiarity with D&D 5e spellcasting (helpful but not required)

## Installation

1. Copy the entire tutorial directory to your skills folder:
   ```bash
   cp -r tutorial-4-magic-system ~/.claude/skills/dnd-magic-system
   ```

2. The skill will be automatically available (Claude reads SKILL.md from the directory)

3. Database tables will be automatically created/extended on first use at `~/.claude/data/dnd-dm.db`

4. Seed the databases on first use:
   ```bash
   cd ~/.claude/skills/dnd-magic-system
   python3 scripts/bestiary.py seed
   python3 scripts/spells.py seed
   ```

## Usage Examples

### Starting Combat with a Wizard

```
You: Let's have Gandalf fight in the training arena

Skill: Let me check Gandalf's equipment and prepare for combat...
       [Checks equipment, equips wizard gear if needed]
       ✓ Equipped Gandalf with wizard starting gear

       Starting combat encounter...

       ⚔️ ✨ TRAINING ARENA ✨ ⚔️

       Gandalf steps into the torch-lit arena, arcane energy
       crackling at his fingertips...

       A Goblin emerges from the shadows!

       Initiative:
         - Gandalf rolled 18 (total: 20)
         - Goblin rolled 5 (total: 7)

       Gandalf strikes first!

       Combat State:
         - Gandalf: 7/7 HP, AC 12, Spell Slots: 2/2
         - Goblin: 4/4 HP, AC 15

       What do you do? (attack, cast spell, flee, surrender)

You: Cast Ray of Frost

Skill: Gandalf extends his hand, frost crystallizing on his fingertips!
       A frigid beam of blue-white light ERUPTS from his palm!

       🎲 Spell Attack: 12 + 5 = 17 vs AC 15
       ❄️ HIT! Damage: 8 cold

       The Goblin freezes solid and shatters!

       💥 VICTORY! 💥

       Spell Slots: 2/2 (Ray of Frost is a cantrip—unlimited!)
```

### Casting a Leveled Spell

```
You: Gandalf casts Magic Missile

Skill: Gandalf thrusts his hands forward! Three glowing darts of
       magical force materialize and streak across the arena!

       ✨ AUTO-HIT! No attack roll needed!

       💫 Three missiles strike for 10 force damage total!

       The Goblin is obliterated!

       💥 VICTORY! 💥

       Spell Slot consumed: 1/2 remaining
```

### Saving Throw Spell

```
You: Cast Burning Hands

Skill: Gandalf spreads his fingers and unleashes a sheet of flames!

       The Goblin must make a DEX save vs DC 13
       🎲 Roll: 4 + 2 = 6
       ❌ FAILED!

       The Goblin takes 10 fire damage and is incinerated!

       💥 VICTORY! 💥

       Spell Slot consumed: 0/2 remaining
```

### Taking a Long Rest

```
You: Gandalf takes a long rest

Skill: ✓ Gandalf takes a long rest
         HP restored to 7
         Spell slots restored

       Gandalf wakes refreshed and ready for more adventures!
       Spell Slots: 2/2
```

## Key Concepts Explained

### Spell Slots (Resource Management)

Spell slots are a limited resource that fuels spellcasting:
- **Cantrips (Level 0)**: Unlimited use, no slots required
- **Leveled Spells (1st level+)**: Consume one slot per cast
- **Restoration**: Long rest fully restores all spell slots
- **Level 1 casters**: 2 spell slots (Wizard, Sorcerer, Cleric, Druid)
- **Non-casters**: 0 spell slots (Fighter, Rogue)

This teaches resource management—a critical game design pattern where players must strategically allocate limited resources.

### Spell Resolution Mechanics

Three different ways spells resolve damage:

1. **Spell Attacks** (like weapon attacks)
   - Caster rolls: d20 + spell attack bonus
   - Compares to target's AC
   - On hit: Roll damage dice
   - Example: Fire Bolt (1d10 fire damage)

2. **Saving Throws** (target resists)
   - Target rolls: d20 + save modifier (DEX, CON, WIS, etc.)
   - Compares to caster's spell save DC
   - On failure: Full damage; On success: Half damage (or none)
   - Example: Burning Hands (3d6 fire damage, DEX save)

3. **Auto-Hit** (guaranteed damage)
   - No attack roll or save needed
   - Damage automatically applied
   - Example: Magic Missile (3 missiles × 1d4+1 force each)

### Spellcasting Stats

Each caster class uses a different ability score:
- **Wizard**: Intelligence (INT)
- **Sorcerer/Warlock**: Charisma (CHA)
- **Cleric/Druid**: Wisdom (WIS)

**Spell attack bonus** = Proficiency + Spellcasting Ability Modifier
**Spell save DC** = 8 + Proficiency + Spellcasting Ability Modifier

For a level 1 Wizard with INT 16:
- Proficiency: +2
- INT modifier: +3
- Spell attack bonus: +5
- Spell save DC: 13

### Modular Data Architecture

Tutorial 4 introduces a **hybrid SQLite + JSON** architecture:

**Reference Data (JSON files)**:
```
data/
└── spells_core.json    # Human-readable spell definitions
```

**Query Layer (SQLite)**:
```sql
CREATE TABLE spells (
    name TEXT PRIMARY KEY,
    level INTEGER,
    school TEXT,
    classes TEXT,           -- JSON array
    stat_block TEXT         -- Full spell data as JSON
)
```

**Why this pattern?**
- ✅ JSON is easy to read, edit, and version control
- ✅ SQLite provides fast queries ("show me all wizard cantrips")
- ✅ Separates reference data (spells from books) from user data (characters)
- ✅ Can add new spells without touching code (just update JSON)
- ✅ Database is derived from authoritative JSON source

### Character Spell Slots (JSON Column)

Instead of adding many columns (`spell_slots_1_max`, `spell_slots_1_current`, `spell_slots_2_max`, etc.), we use a single JSON column:

```sql
ALTER TABLE characters ADD COLUMN spell_slots TEXT;
```

Stored as:
```json
{
  "1": {"max": 2, "current": 1}
}
```

**Advantages:**
- Scales to any number of spell levels without schema changes
- Clean (NULL for non-casters)
- Flexible (can add metadata like "bonus slots" later)
- Matches the pattern already used for equipment/monster stats

## File Structure

```
tutorial-4-magic-system/
├── SKILL.md                    # Claude skill definition
├── README.md                   # This file
├── scripts/
│   ├── roll_dice.py            # (from Tutorial 1)
│   ├── character.py            # Extended with spell slot management
│   ├── equipment.py            # (from Tutorial 3)
│   ├── bestiary.py             # (from Tutorial 3)
│   ├── combat.py               # Extended with spell casting
│   └── spells.py               # NEW: Spell database management
└── data/
    ├── README.md               # JSON format documentation
    └── spells_core.json        # NEW: 10 spells (5 cantrips + 5 level-1)
```

## Available Spells

### Cantrips (Unlimited Use)
- **Fire Bolt** (Wizard/Sorcerer): 1d10 fire damage, spell attack
- **Ray of Frost** (Wizard/Sorcerer): 1d8 cold damage, spell attack
- **Shocking Grasp** (Wizard/Sorcerer): 1d8 lightning damage, spell attack
- **Sacred Flame** (Cleric): 1d8 radiant damage, DEX save
- **Eldritch Blast** (Warlock): 1d10 force damage, spell attack

### Level 1 Spells (Consume Spell Slots)
- **Magic Missile** (Wizard/Sorcerer): 3×(1d4+1) force damage, auto-hit
- **Burning Hands** (Wizard/Sorcerer): 3d6 fire damage, DEX save for half
- **Thunderwave** (Wizard/Sorcerer/Cleric): 2d8 thunder damage, CON save for half
- **Healing Word** (Cleric/Druid): 1d4+modifier healing
- **Shield** (Wizard/Sorcerer): +5 AC until next turn (reaction spell)

## Technical Implementation Notes

### Spell Database Schema

```sql
CREATE TABLE spells (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL UNIQUE,
    level INTEGER NOT NULL,        -- 0 for cantrips, 1-9 for leveled
    school TEXT,                   -- evocation, abjuration, etc.
    classes TEXT,                  -- JSON array: ["wizard", "sorcerer"]
    stat_block TEXT NOT NULL       -- Full spell data as JSON blob
)
```

### Character Schema Extension

```sql
ALTER TABLE characters ADD COLUMN spell_slots TEXT DEFAULT NULL;
```

For casters (level 1):
```json
{"1": {"max": 2, "current": 2}}
```

For non-casters: `NULL`

### Combat Integration

The `combat.py` script now supports a `character-cast` command:

```bash
python3 scripts/combat.py character-cast CHARACTER_NAME "SPELL_NAME" \
    MONSTER_NAME MONSTER_AC MONSTER_HP 'MONSTER_STATS_JSON'
```

Returns JSON with spell results (attack roll, save roll, damage, etc.)

### Spell Slot Functions

**character.py** provides:
- `calculate_spell_slots(class, level)`: Determine slot progression
- `initialize_spell_slots(class, level)`: Create JSON for new characters
- `restore_spell_slots(spell_slots_json)`: Reset to max (long rest)
- `use_spell_slot(spell_slots_json, level)`: Consume a slot

**combat.py** provides:
- `use_spell_slot(character_name, slot_level)`: Consume and update database
- `get_spell_data(spell_name)`: Load spell from database
- `character_cast(...)`: Execute spell with appropriate mechanics

## Extending the System

### Adding New Spells

1. Edit `data/spells_core.json`:
```json
{
  "name": "New Spell",
  "level": 1,
  "school": "evocation",
  "classes": ["wizard"],
  "mechanics": {
    "type": "attack",
    "damage_dice": "2d6",
    "damage_type": "fire"
  }
}
```

2. Re-seed the database:
```bash
python3 scripts/spells.py seed
```

No code changes needed!

### Supporting Higher Levels

To add level 2+ spell slots:
1. Update `calculate_spell_slots()` in `character.py` with progression table
2. Character spell slots JSON will automatically handle: `{"1": {...}, "2": {...}}`

## What Makes This Tutorial Special

Tutorial 4 is a **pivot point** in the series:

**Before Tutorial 4:**
- Tutorials 1-3 focused on building skills from scratch
- Direct, concrete implementations
- Learning by doing

**Tutorial 4 introduces:**
- **Modularity**: Separating data from logic
- **Extensibility**: Adding features without changing code
- **Architecture**: Designing systems that scale
- **Data-driven design**: JSON-configurable behavior

This shift prepares you for building production-quality skills that are maintainable and extensible.

## Common Issues & Solutions

### Spell slots not showing
- Run `python3 scripts/character.py show CHARACTER_NAME`
- If NULL, the character was created before Tutorial 4
- Solution: Create a new character or manually add spell slots

### Spells not found
- Run `python3 scripts/spells.py seed`
- Verify with `python3 scripts/spells.py list`

### Wrong spellcasting ability
- Check character class in database
- Mapping: Wizard=INT, Cleric=WIS, Sorcerer=CHA
- Update `spellcasting_ability_map` in `combat.py` if needed

## Next Steps

After completing Tutorial 4, you'll be ready for:
- **Tutorial 5**: Character progression (leveling, XP, spell scaling)
- **Tutorial 6**: Multi-combatant encounters (party vs multiple monsters)
- **Tutorial 7**: Dungeon exploration (rooms, traps, environmental hazards)

## Learning Outcomes

By completing this tutorial, you can:
- ✅ Design resource management systems
- ✅ Implement hybrid storage architectures (SQLite + JSON)
- ✅ Create data-driven, extensible features
- ✅ Handle conditional game mechanics
- ✅ Manage complex database schemas
- ✅ Build modular, maintainable skills

## Credits

Built as part of the D&D DM Skill tutorial series teaching Claude Code skill development through practical D&D tools.
