#!/usr/bin/env python3
# ABOUTME: Manages D&D combat encounters with turn-based mechanics and spellcasting
# ABOUTME: Handles initiative, attack rolls, damage, spell attacks, saving throws, and victory conditions

import sqlite3
import json
import argparse
import sys
import random
from pathlib import Path

DB_PATH = Path.home() / ".claude" / "data" / "dnd-dm.db"
SKILL_BASE = Path(__file__).parent.parent

def roll_d20():
    """Roll a d20."""
    return random.randint(1, 20)

def roll_dice(dice_str):
    """
    Roll dice from a string like '1d8+2' or '2d6'.
    Returns the total.
    """
    parts = dice_str.split('+')
    dice_part = parts[0]
    modifier = int(parts[1]) if len(parts) > 1 else 0

    # Parse XdY
    num_dice, die_size = map(int, dice_part.split('d'))

    total = sum(random.randint(1, die_size) for _ in range(num_dice)) + modifier
    return total

def get_character_stats(character_name):
    """Load character combat stats from database."""
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("""
        SELECT strength, dexterity, constitution, intelligence, wisdom, charisma,
               hp_current, hp_max, level, class, spell_slots
        FROM characters WHERE name = ?
    """, (character_name,))
    result = cursor.fetchone()
    conn.close()

    if not result:
        return None

    (strength, dexterity, constitution, intelligence, wisdom, charisma,
     hp_current, hp_max, level, char_class, spell_slots_json) = result

    proficiency = 2 + ((level - 1) // 4)

    # Determine spellcasting ability by class
    spellcasting_ability_map = {
        'wizard': intelligence,
        'sorcerer': charisma,
        'warlock': charisma,
        'cleric': wisdom,
        'druid': wisdom,
        'bard': charisma,
        'paladin': charisma,
        'ranger': wisdom
    }

    spellcasting_ability = spellcasting_ability_map.get(char_class.lower())
    spell_mod = (spellcasting_ability - 10) // 2 if spellcasting_ability else None
    spell_attack_bonus = proficiency + spell_mod if spell_mod is not None else None
    spell_save_dc = 8 + proficiency + spell_mod if spell_mod is not None else None

    return {
        "name": character_name,
        "class": char_class,
        "str": strength,
        "str_mod": (strength - 10) // 2,
        "dex": dexterity,
        "dex_mod": (dexterity - 10) // 2,
        "con": constitution,
        "con_mod": (constitution - 10) // 2,
        "int": intelligence,
        "int_mod": (intelligence - 10) // 2,
        "wis": wisdom,
        "wis_mod": (wisdom - 10) // 2,
        "cha": charisma,
        "cha_mod": (charisma - 10) // 2,
        "hp_current": hp_current,
        "hp_max": hp_max,
        "level": level,
        "proficiency": proficiency,
        "spell_attack_bonus": spell_attack_bonus,
        "spell_save_dc": spell_save_dc,
        "spell_slots": json.loads(spell_slots_json) if spell_slots_json else None
    }

def update_character_hp(character_name, new_hp):
    """Update a character's current HP."""
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute(
        "UPDATE characters SET hp_current = ? WHERE name = ?",
        (max(0, new_hp), character_name)
    )
    conn.commit()
    conn.close()

def heal_character(character_name):
    """Fully heal a character (post-combat)."""
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute(
        "UPDATE characters SET hp_current = hp_max WHERE name = ?",
        (character_name,)
    )
    conn.commit()
    conn.close()

def update_character_spell_slots(character_name, spell_slots):
    """Update a character's spell slots."""
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute(
        "UPDATE characters SET spell_slots = ? WHERE name = ?",
        (json.dumps(spell_slots), character_name)
    )
    conn.commit()
    conn.close()

def use_spell_slot(character_name, slot_level):
    """
    Use a spell slot. Returns (success, updated_slots).
    """
    char_stats = get_character_stats(character_name)
    if not char_stats or not char_stats['spell_slots']:
        return False, None

    spell_slots = char_stats['spell_slots']
    level_key = str(slot_level)

    if level_key not in spell_slots:
        return False, spell_slots

    if spell_slots[level_key]['current'] <= 0:
        return False, spell_slots

    spell_slots[level_key]['current'] -= 1
    update_character_spell_slots(character_name, spell_slots)
    return True, spell_slots

def get_spell_data(spell_name):
    """Load spell data from database."""
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute(
        "SELECT name, level, stat_block FROM spells WHERE name = ?",
        (spell_name,)
    )
    result = cursor.fetchone()
    conn.close()

    if not result:
        return None

    name, level, stat_block_json = result
    stat_block = json.loads(stat_block_json)

    return {
        "name": name,
        "level": level,
        **stat_block
    }

def roll_initiative(dex_mod):
    """Roll initiative: d20 + DEX modifier."""
    roll = roll_d20()
    return roll + dex_mod, roll

def attack_roll(attack_bonus, target_ac):
    """
    Make an attack roll.
    Returns: (hit: bool, is_crit: bool, natural_roll: int)
    """
    roll = roll_d20()

    # Natural 20 is always a crit
    if roll == 20:
        return True, True, roll

    # Natural 1 is always a miss
    if roll == 1:
        return False, False, roll

    # Normal attack
    total = roll + attack_bonus
    return total >= target_ac, False, roll

def damage_roll(damage_dice, is_crit=False):
    """
    Roll damage. If crit, roll damage dice twice.
    Returns total damage.
    """
    damage = roll_dice(damage_dice)
    if is_crit:
        damage += roll_dice(damage_dice)
    return damage

def format_combat_state(character, monster, character_ac, turn_num):
    """Format the current combat state for display."""
    output = {
        "turn": turn_num,
        "character": {
            "name": character["name"],
            "hp": f"{character['hp_current']}/{character['hp_max']}",
            "ac": character_ac
        },
        "monster": {
            "name": monster["name"],
            "hp": f"{monster['hp_current']}/{monster['hp_max']}",
            "ac": monster["ac"]
        }
    }
    print(json.dumps(output))

def start_combat(character_name, max_cr):
    """
    Initialize combat encounter.
    Returns JSON with character, monster, initiative, and first turn info.
    """
    # Load character
    character = get_character_stats(character_name)
    if not character:
        print(f"✗ Error: Character '{character_name}' not found", file=sys.stderr)
        return 1

    # Get character AC from equipment
    import subprocess
    try:
        ac_result = subprocess.run(
            [sys.executable, str(Path(__file__).parent / "equipment.py"), "ac", character_name],
            capture_output=True,
            text=True,
            check=True
        )
        character_ac = int(ac_result.stdout.strip())
    except (subprocess.CalledProcessError, ValueError):
        print(f"✗ Error: Could not calculate AC for {character_name}", file=sys.stderr)
        return 1

    # Get character weapon
    try:
        weapon_result = subprocess.run(
            [sys.executable, str(Path(__file__).parent / "equipment.py"), "weapon", character_name],
            capture_output=True,
            text=True,
            check=True
        )
        character_weapon = json.loads(weapon_result.stdout)
    except (subprocess.CalledProcessError, json.JSONDecodeError):
        print(f"✗ Error: Could not get weapon for {character_name}", file=sys.stderr)
        return 1

    # Select monster
    try:
        monster_result = subprocess.run(
            [sys.executable, str(Path(__file__).parent / "bestiary.py"), "select", str(max_cr)],
            capture_output=True,
            text=True,
            check=True
        )
        monster = json.loads(monster_result.stdout)
    except (subprocess.CalledProcessError, json.JSONDecodeError):
        print(f"✗ Error: Could not select monster with CR ≤ {max_cr}", file=sys.stderr)
        return 1

    # Roll monster HP
    monster["hp_max"] = roll_dice(monster["hp_dice"])
    monster["hp_current"] = monster["hp_max"]

    # Roll initiative
    char_init, char_init_roll = roll_initiative(character["dex_mod"])
    monster_dex_mod = (monster["abilities"]["dex"] - 10) // 2
    monster_init, monster_init_roll = roll_initiative(monster_dex_mod)

    # Determine turn order
    char_goes_first = char_init >= monster_init

    # Output combat start state
    output = {
        "character": {
            **character,
            "ac": character_ac,
            "weapon": character_weapon
        },
        "monster": monster,
        "initiative": {
            "character": {"roll": char_init_roll, "total": char_init},
            "monster": {"roll": monster_init_roll, "total": monster_init}
        },
        "first_turn": "character" if char_goes_first else "monster"
    }

    print(json.dumps(output, indent=2))
    return 0

def character_attack(character_name, monster_name, monster_ac, monster_hp):
    """
    Character attacks monster.
    Returns JSON with attack results and updated monster HP.
    """
    # Get character weapon
    import subprocess
    try:
        weapon_result = subprocess.run(
            [sys.executable, str(Path(__file__).parent / "equipment.py"), "weapon", character_name],
            capture_output=True,
            text=True,
            check=True
        )
        weapon = json.loads(weapon_result.stdout)
    except (subprocess.CalledProcessError, json.JSONDecodeError):
        print(f"✗ Error: Could not get weapon for {character_name}", file=sys.stderr)
        return 1

    # Attack roll
    hit, is_crit, natural_roll = attack_roll(weapon["attack_bonus"], monster_ac)

    result = {
        "attacker": character_name,
        "target": monster_name,
        "weapon": weapon["name"],
        "attack_roll": natural_roll,
        "attack_bonus": weapon["attack_bonus"],
        "attack_total": natural_roll + weapon["attack_bonus"],
        "target_ac": monster_ac,
        "hit": hit,
        "crit": is_crit
    }

    if hit:
        damage = damage_roll(weapon["damage"], is_crit)
        result["damage"] = damage
        result["monster_hp_before"] = monster_hp
        result["monster_hp_after"] = max(0, monster_hp - damage)
    else:
        result["damage"] = 0
        result["monster_hp_before"] = monster_hp
        result["monster_hp_after"] = monster_hp

    print(json.dumps(result, indent=2))
    return 0

def character_cast(character_name, spell_name, monster_name, monster_ac, monster_hp, monster_stats_json):
    """
    Character casts a spell at monster.
    Returns JSON with spell results and updated monster HP.
    monster_stats_json should contain monster's ability scores for saving throws.
    """
    # Get character stats
    character = get_character_stats(character_name)
    if not character:
        print(json.dumps({"error": f"Character '{character_name}' not found"}), file=sys.stderr)
        return 1

    # Get spell data
    spell = get_spell_data(spell_name)
    if not spell:
        print(json.dumps({"error": f"Spell '{spell_name}' not found"}), file=sys.stderr)
        return 1

    # Check/consume spell slot (cantrips don't need slots)
    if spell["level"] > 0:
        success, updated_slots = use_spell_slot(character_name, spell["level"])
        if not success:
            print(json.dumps({"error": f"No level {spell['level']} spell slots available"}), file=sys.stderr)
            return 1

    monster_stats = json.loads(monster_stats_json)
    mechanics = spell.get("mechanics", {})
    spell_type = mechanics.get("type")

    result = {
        "caster": character_name,
        "spell": spell_name,
        "spell_level": spell["level"],
        "target": monster_name,
        "spell_type": spell_type
    }

    # Handle different spell types
    if spell_type == "attack":
        # Spell attack roll
        hit, is_crit, natural_roll = attack_roll(character["spell_attack_bonus"], monster_ac)

        result["attack_roll"] = natural_roll
        result["spell_attack_bonus"] = character["spell_attack_bonus"]
        result["attack_total"] = natural_roll + character["spell_attack_bonus"]
        result["target_ac"] = monster_ac
        result["hit"] = hit
        result["crit"] = is_crit

        if hit:
            damage = damage_roll(mechanics["damage_dice"], is_crit)
            result["damage"] = damage
            result["damage_type"] = mechanics["damage_type"]
            result["monster_hp_before"] = monster_hp
            result["monster_hp_after"] = max(0, monster_hp - damage)
        else:
            result["damage"] = 0
            result["monster_hp_before"] = monster_hp
            result["monster_hp_after"] = monster_hp

    elif spell_type == "save":
        # Saving throw
        save_type = mechanics["save_type"]
        save_mod_key = f"{save_type[:3]}_mod"  # e.g., "dex" -> "dex_mod"

        # Calculate monster's save modifier from abilities
        ability_key = save_type[:3]  # dex, con, wis, etc.
        ability_score = monster_stats["abilities"].get(ability_key, 10)
        save_mod = (ability_score - 10) // 2

        save_roll = roll_d20()
        save_total = save_roll + save_mod

        result["save_type"] = save_type.upper()
        result["spell_save_dc"] = character["spell_save_dc"]
        result["save_roll"] = save_roll
        result["save_modifier"] = save_mod
        result["save_total"] = save_total
        result["save_success"] = save_total >= character["spell_save_dc"]

        # Calculate damage
        damage = roll_dice(mechanics["damage_dice"])

        if result["save_success"] and mechanics.get("save_effect") == "half":
            damage = damage // 2

        result["damage"] = damage
        result["damage_type"] = mechanics["damage_type"]
        result["monster_hp_before"] = monster_hp
        result["monster_hp_after"] = max(0, monster_hp - damage)

    elif spell_type == "auto":
        # Auto-hit spells like Magic Missile
        num_missiles = mechanics.get("num_missiles", 1)
        total_damage = 0
        for _ in range(num_missiles):
            total_damage += roll_dice(mechanics["damage_dice"])

        result["num_missiles"] = num_missiles
        result["damage"] = total_damage
        result["damage_type"] = mechanics["damage_type"]
        result["monster_hp_before"] = monster_hp
        result["monster_hp_after"] = max(0, monster_hp - total_damage)

    else:
        print(json.dumps({"error": f"Unsupported spell type: {spell_type}"}), file=sys.stderr)
        return 1

    print(json.dumps(result, indent=2))
    return 0

def monster_attack(monster_name, monster_attack_bonus, monster_damage, character_name, character_ac):
    """
    Monster attacks character.
    Returns JSON with attack results and updated character HP.
    """
    # Get current character HP
    character = get_character_stats(character_name)
    if not character:
        print(f"✗ Error: Character '{character_name}' not found", file=sys.stderr)
        return 1

    # Attack roll
    hit, is_crit, natural_roll = attack_roll(monster_attack_bonus, character_ac)

    result = {
        "attacker": monster_name,
        "target": character_name,
        "attack_roll": natural_roll,
        "attack_bonus": monster_attack_bonus,
        "attack_total": natural_roll + monster_attack_bonus,
        "target_ac": character_ac,
        "hit": hit,
        "crit": is_crit
    }

    if hit:
        damage = damage_roll(monster_damage, is_crit)
        result["damage"] = damage
        result["character_hp_before"] = character["hp_current"]
        new_hp = max(0, character["hp_current"] - damage)
        result["character_hp_after"] = new_hp

        # Update character HP
        update_character_hp(character_name, new_hp)
    else:
        result["damage"] = 0
        result["character_hp_before"] = character["hp_current"]
        result["character_hp_after"] = character["hp_current"]

    print(json.dumps(result, indent=2))
    return 0

def end_combat(character_name, outcome):
    """
    End combat and apply post-combat effects.
    outcome: 'victory', 'defeat', 'fled'
    """
    if outcome == "victory":
        heal_character(character_name)
        print(f"✓ Victory! {character_name} has been fully healed.")
    elif outcome == "defeat":
        print(f"✗ Defeat! {character_name} was defeated.")
    elif outcome == "fled":
        heal_character(character_name)
        print(f"✓ {character_name} fled the battle and has been healed.")

    return 0

def main():
    parser = argparse.ArgumentParser(description="D&D Combat System")
    subparsers = parser.add_subparsers(dest="command", help="Command to execute")

    # Start combat
    start_parser = subparsers.add_parser("start", help="Start a combat encounter")
    start_parser.add_argument("character_name", help="Character name")
    start_parser.add_argument("max_cr", type=float, help="Maximum monster CR")

    # Character attack
    char_attack_parser = subparsers.add_parser("character-attack", help="Character attacks")
    char_attack_parser.add_argument("character_name", help="Character name")
    char_attack_parser.add_argument("monster_name", help="Monster name")
    char_attack_parser.add_argument("monster_ac", type=int, help="Monster AC")
    char_attack_parser.add_argument("monster_hp", type=int, help="Current monster HP")

    # Character cast spell
    char_cast_parser = subparsers.add_parser("character-cast", help="Character casts a spell")
    char_cast_parser.add_argument("character_name", help="Character name")
    char_cast_parser.add_argument("spell_name", help="Spell name")
    char_cast_parser.add_argument("monster_name", help="Monster name")
    char_cast_parser.add_argument("monster_ac", type=int, help="Monster AC")
    char_cast_parser.add_argument("monster_hp", type=int, help="Current monster HP")
    char_cast_parser.add_argument("monster_stats", help="Monster stats as JSON (for saving throws)")

    # Monster attack
    monster_attack_parser = subparsers.add_parser("monster-attack", help="Monster attacks")
    monster_attack_parser.add_argument("monster_name", help="Monster name")
    monster_attack_parser.add_argument("monster_attack_bonus", type=int, help="Monster attack bonus")
    monster_attack_parser.add_argument("monster_damage", help="Monster damage dice (e.g., 1d6+2)")
    monster_attack_parser.add_argument("character_name", help="Character name")
    monster_attack_parser.add_argument("character_ac", type=int, help="Character AC")

    # End combat
    end_parser = subparsers.add_parser("end", help="End combat encounter")
    end_parser.add_argument("character_name", help="Character name")
    end_parser.add_argument("outcome", choices=["victory", "defeat", "fled"], help="Combat outcome")

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        return 1

    if args.command == "start":
        return start_combat(args.character_name, args.max_cr)
    elif args.command == "character-attack":
        return character_attack(args.character_name, args.monster_name, args.monster_ac, args.monster_hp)
    elif args.command == "character-cast":
        return character_cast(args.character_name, args.spell_name, args.monster_name,
                            args.monster_ac, args.monster_hp, args.monster_stats)
    elif args.command == "monster-attack":
        return monster_attack(args.monster_name, args.monster_attack_bonus, args.monster_damage,
                            args.character_name, args.character_ac)
    elif args.command == "end":
        return end_combat(args.character_name, args.outcome)

    return 0

if __name__ == "__main__":
    random.seed()
    sys.exit(main())
