#!/usr/bin/env python3
# ABOUTME: D&D dice roller script that parses notation and returns results
# ABOUTME: Supports: d20, 2d6+3, advantage/disadvantage (via flags), drop lowest

import sys
import random
import re
import argparse


def parse_dice_notation(notation):
    """Parse dice notation like '2d6+3' or 'd20' into components."""
    notation = notation.lower().strip()

    # Parse XdY+Z or XdY-Z format
    match = re.match(r'^(\d*)d(\d+)([+-]\d+)?$', notation)
    if not match:
        return None

    num_dice = int(match.group(1)) if match.group(1) else 1
    die_size = int(match.group(2))
    modifier = int(match.group(3)) if match.group(3) else 0

    return {
        'num_dice': num_dice,
        'die_size': die_size,
        'modifier': modifier
    }


def roll_dice(num_dice, die_size):
    """Roll specified number of dice with given size."""
    return [random.randint(1, die_size) for _ in range(num_dice)]


def main():
    parser = argparse.ArgumentParser(description='Roll D&D dice')
    parser.add_argument('notation', help='Dice notation (e.g., d20, 2d6+3)')
    parser.add_argument('--advantage', action='store_true', help='Roll with advantage (roll twice, take higher)')
    parser.add_argument('--disadvantage', action='store_true', help='Roll with disadvantage (roll twice, take lower)')
    parser.add_argument('--drop-lowest', action='store_true', help='Drop the lowest die (e.g., for 4d6 drop lowest)')

    args = parser.parse_args()

    parsed = parse_dice_notation(args.notation)

    if not parsed:
        print(f"Error: Invalid dice notation '{args.notation}'")
        print("Valid formats: d20, 2d6+3, 1d8-1")
        sys.exit(1)

    num_dice = parsed['num_dice']
    die_size = parsed['die_size']
    modifier = parsed['modifier']

    # Handle advantage/disadvantage
    if args.advantage or args.disadvantage:
        roll1 = roll_dice(1, die_size)[0]
        roll2 = roll_dice(1, die_size)[0]
        result = max(roll1, roll2) if args.advantage else min(roll1, roll2)
        advantage_type = "advantage" if args.advantage else "disadvantage"

        print(f"Rolling d{die_size} with {advantage_type}...")
        print(f"[{roll1}] [{roll2}] ({advantage_type}) = {result}")
        return

    # Handle drop lowest
    if args.drop_lowest:
        rolls = roll_dice(num_dice, die_size)
        dropped = min(rolls)
        remaining = rolls.copy()
        remaining.remove(dropped)
        total = sum(remaining)

        print(f"Rolling {num_dice}d{die_size}, dropping lowest...")
        print(f"{rolls} → Dropped [{dropped}]")
        print(f"{remaining} = {total}")
        return

    # Standard roll
    rolls = roll_dice(num_dice, die_size)
    total = sum(rolls) + modifier

    dice_str = f"{num_dice}d{die_size}" if num_dice > 1 else f"d{die_size}"
    modifier_str = f"{modifier:+d}" if modifier != 0 else ""

    print(f"Rolling {dice_str}{modifier_str}...")

    if num_dice == 1:
        if modifier != 0:
            print(f"[{rolls[0]}] {modifier:+d} = {total}")
        else:
            print(f"[{rolls[0]}] = {total}")
    else:
        rolls_str = str(rolls).replace('[', '').replace(']', '')
        if modifier != 0:
            print(f"[{rolls_str}] {modifier:+d} = {total}")
        else:
            print(f"[{rolls_str}] = {total}")


if __name__ == '__main__':
    main()
