#!/usr/bin/env python3
# ABOUTME: Manages the spell database for D&D spells
# ABOUTME: Handles CRUD operations for spells with level and class filtering

import sqlite3
import json
import argparse
import sys
from pathlib import Path

DB_PATH = Path.home() / ".claude" / "data" / "dnd-dm.db"
SKILL_BASE = Path(__file__).parent.parent

def init_db():
    """Initialize the database and create spells table if it doesn't exist."""
    DB_PATH.parent.mkdir(parents=True, exist_ok=True)
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS spells (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL UNIQUE,
            level INTEGER NOT NULL,
            school TEXT,
            classes TEXT,
            stat_block TEXT NOT NULL
        )
    """)

    conn.commit()
    conn.close()

def add_spell(name, level, school, classes, stat_block):
    """Add a spell to the database."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    try:
        # Validate JSON
        json.loads(stat_block)

        cursor.execute(
            "INSERT INTO spells (name, level, school, classes, stat_block) VALUES (?, ?, ?, ?, ?)",
            (name, level, school, json.dumps(classes) if isinstance(classes, list) else classes, stat_block)
        )
        conn.commit()
        print(f"✓ Added {name} (Level {level}) to spells")
        return 0
    except json.JSONDecodeError:
        print(f"✗ Error: Invalid JSON in stat_block", file=sys.stderr)
        return 1
    except sqlite3.IntegrityError:
        print(f"✗ Error: Spell '{name}' already exists", file=sys.stderr)
        return 1
    finally:
        conn.close()

def show_spell(name):
    """Display a spell's full details."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("SELECT name, level, school, classes, stat_block FROM spells WHERE name = ?", (name,))
    result = cursor.fetchone()
    conn.close()

    if not result:
        print(f"✗ Error: Spell '{name}' not found", file=sys.stderr)
        return 1

    name, level, school, classes_json, stat_block_json = result
    classes = json.loads(classes_json)
    stat_block = json.loads(stat_block_json)

    level_str = "Cantrip" if level == 0 else f"Level {level}"
    print(f"\n{name} ({level_str} {school.capitalize()})")
    print("=" * 40)
    print(f"Classes: {', '.join(classes)}")
    print(f"Casting Time: {stat_block.get('casting_time', 'Unknown')}")
    print(f"Range: {stat_block.get('range', 'Unknown')}")
    print(f"Components: {', '.join(stat_block.get('components', []))}")
    print(f"Duration: {stat_block.get('duration', 'Unknown')}")
    print(f"\n{stat_block.get('description', 'No description')}")

    mechanics = stat_block.get('mechanics', {})
    if mechanics:
        print(f"\nMechanics:")
        print(f"  Type: {mechanics.get('type', 'Unknown')}")
        if 'damage_dice' in mechanics:
            print(f"  Damage: {mechanics['damage_dice']} {mechanics.get('damage_type', '')} damage")
        if 'save_type' in mechanics:
            print(f"  Save: {mechanics['save_type'].upper()} save")

    return 0

def list_spells(level=None, char_class=None):
    """List all spells, optionally filtered by level and/or class."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    query = "SELECT name, level, school, classes FROM spells"
    params = []
    conditions = []

    if level is not None:
        conditions.append("level = ?")
        params.append(level)

    if conditions:
        query += " WHERE " + " AND ".join(conditions)

    query += " ORDER BY level, name"

    cursor.execute(query, params)
    results = cursor.fetchall()
    conn.close()

    if not results:
        print("No spells found")
        return 0

    # Filter by class if specified (since classes is JSON array)
    if char_class:
        filtered_results = []
        for name, lvl, school, classes_json in results:
            classes = json.loads(classes_json)
            if char_class.lower() in [c.lower() for c in classes]:
                filtered_results.append((name, lvl, school, classes_json))
        results = filtered_results

    if not results:
        print(f"No spells found for {char_class}")
        return 0

    # Group by level
    current_level = None
    for name, lvl, school, classes_json in results:
        if lvl != current_level:
            current_level = lvl
            level_str = "Cantrips" if lvl == 0 else f"Level {lvl} Spells"
            print(f"\n{level_str}:")

        classes = json.loads(classes_json)
        print(f"  • {name} ({school}) - {', '.join(classes)}")

    return 0

def get_spell(name):
    """Get a spell's full data as JSON (for use by combat system)."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("SELECT name, level, school, classes, stat_block FROM spells WHERE name = ?", (name,))
    result = cursor.fetchone()
    conn.close()

    if not result:
        print(json.dumps({"error": f"Spell '{name}' not found"}), file=sys.stderr)
        return 1

    name, level, school, classes_json, stat_block_json = result
    classes = json.loads(classes_json)
    stat_block = json.loads(stat_block_json)

    output = {
        "name": name,
        "level": level,
        "school": school,
        "classes": classes,
        **stat_block
    }
    print(json.dumps(output))
    return 0

def seed_spells():
    """Seed the spell database from spells_core.json."""
    init_db()

    # Load spells from JSON file
    spells_file = SKILL_BASE / "assets" / "data" / "spells_core.json"

    if not spells_file.exists():
        print(f"✗ Error: {spells_file} not found", file=sys.stderr)
        return 1

    with open(spells_file) as f:
        data = json.load(f)

    spells = data.get("spells", [])
    if not spells:
        print("✗ Error: No spells found in JSON file", file=sys.stderr)
        return 1

    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    seeded_count = 0
    for spell in spells:
        try:
            cursor.execute(
                "INSERT INTO spells (name, level, school, classes, stat_block) VALUES (?, ?, ?, ?, ?)",
                (
                    spell["name"],
                    spell["level"],
                    spell["school"],
                    json.dumps(spell["classes"]),
                    json.dumps(spell)
                )
            )
            seeded_count += 1
        except sqlite3.IntegrityError:
            # Spell already exists, skip
            pass

    conn.commit()
    conn.close()

    if seeded_count > 0:
        print(f"✓ Seeded {seeded_count} spells into database")
    else:
        print("Spells already seeded")

    return 0

def main():
    parser = argparse.ArgumentParser(description="Manage D&D spell database")
    subparsers = parser.add_subparsers(dest="command", help="Command to execute")

    # Add command
    add_parser = subparsers.add_parser("add", help="Add a spell to the database")
    add_parser.add_argument("name", help="Spell name")
    add_parser.add_argument("level", type=int, help="Spell level (0 for cantrip)")
    add_parser.add_argument("school", help="School of magic")
    add_parser.add_argument("classes", help="Classes that can cast (comma-separated)")
    add_parser.add_argument("stat_block", help="JSON stat block")

    # Show command
    show_parser = subparsers.add_parser("show", help="Show a spell's details")
    show_parser.add_argument("name", help="Spell name")

    # List command
    list_parser = subparsers.add_parser("list", help="List all spells")
    list_parser.add_argument("--level", type=int, help="Filter by spell level")
    list_parser.add_argument("--class", dest="char_class", help="Filter by class")

    # Get command (for JSON output)
    get_parser = subparsers.add_parser("get", help="Get spell data as JSON")
    get_parser.add_argument("name", help="Spell name")

    # Seed command
    subparsers.add_parser("seed", help="Seed spell database from spells_core.json")

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        return 1

    if args.command == "add":
        classes_list = [c.strip() for c in args.classes.split(",")]
        return add_spell(args.name, args.level, args.school, classes_list, args.stat_block)
    elif args.command == "show":
        return show_spell(args.name)
    elif args.command == "list":
        return list_spells(args.level, args.char_class)
    elif args.command == "get":
        return get_spell(args.name)
    elif args.command == "seed":
        return seed_spells()

    return 0

if __name__ == "__main__":
    sys.exit(main())
