# Tutorial 5: Character Progression

Learn to build skills with **threshold triggers**, **progressive complexity**, and **calculated growth** by implementing a D&D 5e character advancement system with XP tracking and automatic leveling.

## What You'll Learn

### Skill-Building Concepts

1. **Threshold Triggers** - Automatic events that fire when values cross specific thresholds
2. **Progressive Complexity** - Features that unlock gradually as a system advances
3. **Calculated Growth** - Growth following deterministic formulas rather than arbitrary values

### D&D Features Implemented

- Experience Point (XP) tracking
- Automatic level-ups when XP thresholds are reached
- HP increases on level-up (based on class hit dice)
- Proficiency bonus progression
- Spell slot advancement for casters
- Full integration with Tutorial 4's combat system

## Prerequisites

Complete **Tutorial 4: Magic System** first. This tutorial extends the combat system with character progression mechanics.

## Installation

Copy the tutorial directory to your Claude skills folder:

```bash
cp -r tutorial-5-character-progression ~/.claude/skills/tutorial-5
```

## Quick Start

### 1. Create a Character

```bash
python3 ~/.claude/skills/tutorial-5/scripts/character.py create "Aria" wizard \
  --str 8 --dex 14 --con 12 --int 16 --wis 13 --cha 10
```

### 2. Seed Databases

```bash
python3 ~/.claude/skills/tutorial-5/scripts/bestiary.py seed
python3 ~/.claude/skills/tutorial-5/scripts/spells.py seed
```

### 3. Equip Character

```bash
python3 ~/.claude/skills/tutorial-5/scripts/equipment.py equip Aria wizard
```

### 4. Enter the Arena and Fight!

```bash
# Start combat (MAX_CR = level/4 = 0.25 for level 1)
python3 ~/.claude/skills/tutorial-5/scripts/combat.py start Aria 0.25

# Character attacks or casts spells...
python3 ~/.claude/skills/tutorial-5/scripts/combat.py character-cast Aria "Fire Bolt" Goblin 15 9 '{"dex": 14, ...}'

# Monster attacks...
python3 ~/.claude/skills/tutorial-5/scripts/combat.py monster-attack Goblin 4 "1d6+2" Aria 12

# End combat with XP award
python3 ~/.claude/skills/tutorial-5/scripts/combat.py end Aria victory --monster-cr 0.25
```

**Result:**
```
✓ Victory! Aria has been fully healed.
✓ Awarded 50 XP to Aria
  Total XP: 0 → 50
  XP to level 2: 250 more needed
```

### 5. Level Up!

After 6 victories against CR 1/4 Goblins (300 XP total):

```
✓ Awarded 50 XP to Aria
  Total XP: 250 → 300

🎉 LEVEL UP! Aria is now level 2!
  Levels gained: 1
  HP increased: 7 → 13 (+6)
  Proficiency bonus: +2
  XP to next level: 600
```

## Tutorial Structure

```
tutorial-5-character-progression/
├── SKILL.md                     # Lean skill file with workflow
├── README.md                    # This file
├── scripts/
│   ├── roll_dice.py            # From Tutorial 1
│   ├── character.py            # Extended with XP tracking
│   ├── bestiary.py             # From Tutorial 3
│   ├── equipment.py            # From Tutorial 3
│   ├── spells.py               # From Tutorial 4
│   ├── combat.py               # Extended with XP awards
│   └── progression.py          # NEW: Leveling logic
├── references/
│   └── progression-rules.md    # Complete XP tables and rules
├── assets/
│   └── data/
│       ├── spells_core.json   # Spell database
│       └── xp_tables.json     # XP thresholds and CR awards
└── examples/
    └── (example combat sessions)
```

## Core Mechanics

### Experience Points (XP)

XP is awarded after combat victories based on monster Challenge Rating:

| CR    | XP Award | Example Monster |
|-------|----------|-----------------|
| 0     | 10       | Rat             |
| 1/4   | 50       | Goblin          |
| 1/2   | 100      | Orc             |
| 1     | 200      | Dire Wolf       |
| 2     | 450      | Ogre            |

### Level-Up Thresholds

Characters automatically level up when XP reaches these thresholds:

| Level | Total XP | XP from Prev | Goblin Fights (50 XP) |
|-------|----------|--------------|------------------------|
| 1     | 0        | —            | Starting level         |
| 2     | 300      | +300         | 6 fights               |
| 3     | 900      | +600         | 12 more fights         |
| 4     | 2,700    | +1,800       | 36 more fights         |
| 5     | 6,500    | +3,800       | 76 more fights         |

### Level-Up Benefits

Each level automatically grants:

**1. Hit Points**
- Gain average hit die + CON modifier
- Wizard (d6): +4 + CON per level
- Fighter (d10): +6 + CON per level

**2. Proficiency Bonus**
- Levels 1-4: +2
- Levels 5-8: +3
- Levels 9-12: +4
- Levels 13-16: +5
- Levels 17-20: +6

**3. Spell Slots (Casters)**
- Level 1: 2 × 1st-level
- Level 2: 3 × 1st-level
- Level 3: 4 × 1st-level, 2 × 2nd-level
- Level 4: 4 × 1st-level, 3 × 2nd-level
- Level 5: 4 × 1st-level, 3 × 2nd-level, 2 × 3rd-level

## Key Scripts

### progression.py (NEW)

The star of Tutorial 5! Handles XP awards and automatic leveling.

**Award XP:**
```bash
python3 scripts/progression.py award CHARACTER_NAME XP_AMOUNT
```

**View XP Table:**
```bash
python3 scripts/progression.py xp-table
```

**View CR XP Awards:**
```bash
python3 scripts/progression.py cr-xp
```

### character.py (Extended)

Now displays XP and progress to next level:

```bash
python3 scripts/character.py show Aria

# Output:
# Aria (wizard, Level 1)
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# XP: 50/300 (need 250 more)
# ...
```

### combat.py (Extended)

New `--monster-cr` flag on `end` command awards XP:

```bash
python3 scripts/combat.py end CHARACTER_NAME victory --monster-cr 0.25
```

## Example Session

Here's a complete progression from level 1 to level 2:

```bash
# Create character
python3 scripts/character.py create "Theron" fighter \
  --str 16 --dex 12 --con 14 --int 10 --wis 13 --cha 8

# Equip and seed
python3 scripts/equipment.py equip Theron fighter
python3 scripts/bestiary.py seed
python3 scripts/spells.py seed

# Fight #1 - Goblin
python3 scripts/combat.py start Theron 0.25
# ... combat happens ...
python3 scripts/combat.py end Theron victory --monster-cr 0.25
# ✓ Awarded 50 XP (50/300)

# Fight #2 - Goblin
python3 scripts/combat.py start Theron 0.25
# ... combat ...
python3 scripts/combat.py end Theron victory --monster-cr 0.25
# ✓ Awarded 50 XP (100/300)

# Fight #3 - Goblin
python3 scripts/combat.py start Theron 0.25
# ... combat ...
python3 scripts/combat.py end Theron victory --monster-cr 0.25
# ✓ Awarded 50 XP (150/300)

# Fight #4 - Goblin
python3 scripts/combat.py start Theron 0.25
# ... combat ...
python3 scripts/combat.py end Theron victory --monster-cr 0.25
# ✓ Awarded 50 XP (200/300)

# Fight #5 - Goblin
python3 scripts/combat.py start Theron 0.25
# ... combat ...
python3 scripts/combat.py end Theron victory --monster-cr 0.25
# ✓ Awarded 50 XP (250/300)

# Fight #6 - LEVEL UP!
python3 scripts/combat.py start Theron 0.25
# ... combat ...
python3 scripts/combat.py end Theron victory --monster-cr 0.25

# Output:
# ✓ Awarded 50 XP to Theron
#   Total XP: 250 → 300
#
# 🎉 LEVEL UP! Theron is now level 2!
#   Levels gained: 1
#   HP increased: 13 → 22 (+9)
#   Proficiency bonus: +2
#   XP to next level: 600

# Check new stats
python3 scripts/character.py show Theron
# Theron (fighter, Level 2)
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# XP: 300/900 (need 600 more)
# HP: 22/22
# ...
```

## Skill Concepts in Action

### 1. Threshold Triggers

**What it is:** Automatic events that fire when a value crosses a specific threshold.

**In Tutorial 5:**
```python
# In progression.py
while new_xp >= XP_THRESHOLDS[new_level]:
    new_level += 1
    levels_gained += 1
```

When XP reaches 300, the system **automatically** triggers a level-up. No manual intervention required. The character gains HP, proficiency increases, spell slots refresh—all computed and applied instantly.

**Real-World Applications:**
- **Alert systems**: Temperature > 100°F → trigger cooling system
- **Subscription tiers**: API usage > 1000 calls → upgrade to next tier
- **Achievement systems**: Score crosses 10,000 → unlock "Master" badge
- **Inventory management**: Stock < 10 units → trigger reorder

**Key Insight:** Threshold triggers automate state transitions, reducing manual work and ensuring consistency.

### 2. Progressive Complexity

**What it is:** Features unlock gradually as the system advances, not all at once.

**In Tutorial 5:**
- **Level 1**: Basic combat with 2 spell slots
- **Level 2**: +HP, +1 spell slot (3 total)
- **Level 3**: +HP, unlock 2nd-level spells (major power spike!)
- **Level 4**: +HP, +1 more 1st-level slot
- **Level 5**: +HP, +proficiency bonus, unlock 3rd-level spells

Each level adds incremental improvements with occasional big jumps (new spell levels). This creates a satisfying sense of growth and anticipation.

**Real-World Applications:**
- **SaaS products**: Basic → Pro → Enterprise (features unlock with tier)
- **Educational platforms**: Basic lessons → Advanced modules → Expert content
- **Game design**: Start with simple mechanics, unlock complexity over time
- **User onboarding**: Show basic features first, advanced features after proficiency

**Key Insight:** Progressive complexity prevents overwhelm and creates sustained engagement through gradual unlocks.

### 3. Calculated Growth

**What it is:** Growth follows deterministic formulas, not arbitrary values.

**In Tutorial 5:**
```python
# HP increase formula
def calculate_hp_increase(character_class, constitution_score):
    hit_dice = {'wizard': 6, 'fighter': 10, ...}
    hd = hit_dice[character_class]
    con_mod = (constitution_score - 10) // 2
    avg_roll = (hd // 2) + 1
    return avg_roll + con_mod

# Proficiency bonus formula
proficiency = 2 + ((level - 1) // 4)

# Spell slots formula
slots = lookup_table(character_class, level)
```

Every stat increase is **predictable** based on class, level, and ability scores. Players can calculate their level 10 stats at level 1. No surprises, no randomness—pure math.

**Real-World Applications:**
- **Pricing calculators**: base_price + (users × per_user_rate)
- **Compound interest**: principal × (1 + rate)^time
- **Performance metrics**: response_time = base_latency + (load × scaling_factor)
- **Capacity planning**: servers_needed = ceiling(traffic / server_capacity)

**Key Insight:** Calculated growth provides predictability and transparency, building user trust and enabling planning.

## Combining the Concepts

The magic happens when all three work together:

1. **XP accumulates** (simple counter)
2. **Threshold trigger** fires when XP >= 300
3. **Progressive complexity** unlocks new spell slots at level 3
4. **Calculated growth** determines exactly how much HP increases

This creates a **self-managing progression system** that:
- Requires no manual intervention
- Provides clear feedback
- Follows consistent rules
- Scales from level 1 to 20

## Extension Ideas

Once you've completed this tutorial, try extending it:

1. **Ability Score Improvements (ASI)**
   - Grant +2 to one stat or +1 to two stats at levels 4, 8, 12, 16, 19
   - Recalculate HP, AC, spell DCs when stats change

2. **Multi-Classing**
   - Allow characters to take levels in multiple classes
   - Complex spell slot calculations
   - Track levels per class separately

3. **Combat History Tracking**
   - Create a `combat_history` table
   - Track victories, defeats, monsters defeated
   - Generate combat statistics and reports

4. **Class Features**
   - Level 2: Class-specific abilities
   - Level 3: Subclass choice
   - Level 5: Extra Attack for martials

5. **Milestone Leveling**
   - Alternative to XP: level up after quest completion
   - Track narrative progress instead of numerical XP

## What's Next?

**Tutorial 6: Multi-Combatant Encounters** will teach:
- **Collection Management** - Managing multiple characters and monsters
- **Priority Queues** - Turn order with dynamic initiative
- **Complex State Synchronization** - Keeping party and enemy states in sync
- **Targeting Logic** - Area-of-effect spells and multi-target attacks

You'll build party-based combat with multiple PCs and monsters, initiative tracking, and AOE spells.

## Troubleshooting

### "No such column: xp" Error

You have an old database from Tutorial 4. Delete and recreate:

```bash
rm ~/.claude/data/dnd-dm.db
# Then create a new character
```

### Level-up Not Triggering

Check XP threshold. Level 2 requires exactly **300 XP**:

```bash
python3 scripts/progression.py xp-table  # View thresholds
python3 scripts/character.py show Aria   # Check current XP
```

### Character Not Leveling Past Level 1

Make sure you're passing `--monster-cr` when ending combat:

```bash
# ❌ Wrong - no XP awarded
python3 scripts/combat.py end Aria victory

# ✅ Correct - XP awarded for CR 0.25 Goblin
python3 scripts/combat.py end Aria victory --monster-cr 0.25
```

## Files Overview

### Core Scripts

- **`progression.py`** (NEW) - XP awards, automatic leveling, threshold triggers
- **`character.py`** - Extended with XP field and XP display
- **`combat.py`** - Extended with `--monster-cr` flag for XP awards

### Data Files

- **`assets/data/xp_tables.json`** - XP thresholds and CR awards
- **`assets/data/spells_core.json`** - Spell database (from Tutorial 4)

### Documentation

- **`references/progression-rules.md`** - Complete XP tables, formulas, strategy guide
- **Tutorial 4's references** - Combat narrative, spell mechanics, D&D rules (still applicable)

## Summary

Tutorial 5 teaches you to build **self-managing progression systems** using:

1. **Threshold Triggers** - Automatic level-ups when XP crosses thresholds
2. **Progressive Complexity** - Features unlock gradually (spell slots, proficiency)
3. **Calculated Growth** - Deterministic formulas for stat increases

These concepts apply far beyond D&D—use them to build subscription tiers, achievement systems, alert triggers, and any system that grows or changes based on accumulated metrics.

Now go forth and level up! 🎉
