# Character Progression and Leveling Rules

## Overview

Tutorial 5 introduces character advancement through experience points (XP) and leveling. This system teaches progressive complexity (features unlock with level), threshold triggers (automatic events at milestones), and calculated growth (stat increases based on formulas).

## Experience Points (XP)

### How XP Works

Experience Points represent a character's accumulated combat experience and training. As characters earn XP, they automatically level up when reaching specific thresholds.

**Key Concepts:**
- XP is awarded after combat victories
- XP accumulates over time (never resets)
- Reaching an XP threshold triggers automatic level-up
- Characters can level up multiple times from a single large XP award

### XP Thresholds by Level

Based on D&D 5th Edition Player's Handbook (p.15):

| Level | Total XP Required | XP from Previous Level |
|-------|------------------|------------------------|
| 1     | 0                | Starting level         |
| 2     | 300              | +300                   |
| 3     | 900              | +600                   |
| 4     | 2,700            | +1,800                 |
| 5     | 6,500            | +3,800                 |
| 6     | 14,000           | +7,500                 |
| 7     | 23,000           | +9,000                 |
| 8     | 34,000           | +11,000                |
| 9     | 48,000           | +14,000                |
| 10    | 64,000           | +16,000                |
| 11    | 85,000           | +21,000                |
| 12    | 100,000          | +15,000                |
| 13    | 120,000          | +20,000                |
| 14    | 140,000          | +20,000                |
| 15    | 165,000          | +25,000                |
| 16    | 195,000          | +30,000                |
| 17    | 225,000          | +30,000                |
| 18    | 265,000          | +40,000                |
| 19    | 305,000          | +40,000                |
| 20    | 355,000          | +50,000                |

**Progression Note:** XP requirements increase exponentially. Early levels come quickly (level 1→2 requires only 300 XP), while later levels take much longer (level 19→20 requires 50,000 XP).

### XP Awards by Monster CR

Characters earn XP by defeating monsters in the training arena. XP is based on the monster's Challenge Rating (CR):

| Challenge Rating | XP Award | Example Monsters    |
|------------------|----------|---------------------|
| CR 0             | 10 XP    | Rat                 |
| CR 1/8           | 25 XP    | —                   |
| CR 1/4           | 50 XP    | Goblin, Skeleton    |
| CR 1/2           | 100 XP   | Orc                 |
| CR 1             | 200 XP   | Dire Wolf           |
| CR 2             | 450 XP   | Ogre                |
| CR 3             | 700 XP   | Manticore           |
| CR 4             | 1,100 XP | Ettin               |
| CR 5             | 1,800 XP | Hill Giant          |

**Note:** In Tutorial 5, XP is only awarded for victories. Fleeing or defeat grants no XP.

### XP Award Examples

**Example 1: Level 1 Fighter vs. Goblin (CR 1/4)**
- Goblin defeated: +50 XP
- Character XP: 0 → 50 XP
- Progress to level 2: 50/300 (16.7%)
- Battles to level 2: ~6 more Goblins

**Example 2: Level 1 Wizard vs. Rat (CR 0)**
- Rat defeated: +10 XP
- Character XP: 0 → 10 XP
- Progress to level 2: 10/300 (3.3%)
- Battles to level 2: 29 more Rats

**Example 3: Multiple Victories**
- Start: Level 1, 0 XP
- Defeat Goblin (CR 1/4): +50 XP → 50/300
- Defeat Goblin (CR 1/4): +50 XP → 100/300
- Defeat Goblin (CR 1/4): +50 XP → 150/300
- Defeat Goblin (CR 1/4): +50 XP → 200/300
- Defeat Goblin (CR 1/4): +50 XP → 250/300
- Defeat Goblin (CR 1/4): +50 XP → 300/300 → **LEVEL UP to 2!**

## Leveling Up

### Automatic Level-Up System

When a character's total XP meets or exceeds the threshold for the next level, they **automatically level up**. This is a **threshold trigger** - the system detects the milestone and applies benefits immediately.

**Level-Up Sequence:**
1. Character defeats monster
2. XP awarded via `progression.py award`
3. System checks: `current_xp >= xp_threshold[next_level]`?
4. If yes: Automatic level-up with all benefits applied
5. Character immediately gains: HP, proficiency bonus, spell slots

**Multiple Level-Ups:**
If a single XP award pushes a character across multiple thresholds, they gain all levels at once:

```
Example: Level 1 character (0 XP) somehow defeats CR 5 monster
- Award: +1,800 XP
- Check level 2 (300 XP): 1,800 >= 300 ✓ → Level up!
- Check level 3 (900 XP): 1,800 >= 900 ✓ → Level up!
- Check level 4 (2,700 XP): 1,800 >= 2,700 ✗ → Stop
- Result: Character is now level 3 with 1,800 XP
```

### Level-Up Benefits

Each level grants specific benefits based on D&D 5e rules:

#### 1. Hit Point Increase

Characters gain HP every level based on their class's Hit Die:

**Hit Dice by Class:**
- d6: Wizard, Sorcerer (fragile spellcasters)
- d8: Rogue, Bard, Cleric, Druid, Monk, Warlock (moderate)
- d10: Fighter, Paladin, Ranger (martial warriors)
- d12: Barbarian (toughest)

**HP Calculation:**
- **Level 1**: Max Hit Die + CON modifier
- **Each additional level**: Average roll + CON modifier
  - Average roll = (Hit Die / 2) + 1
  - Example: d8 → average = 5

**Example - Wizard (d6, CON 14 [+2]):**
- Level 1: 6 (max d6) + 2 (CON) = **8 HP**
- Level 2: +4 (avg d6) + 2 (CON) = **14 HP total** (+6)
- Level 3: +4 + 2 = **20 HP total** (+6)
- Level 4: +4 + 2 = **26 HP total** (+6)

**Example - Fighter (d10, CON 16 [+3]):**
- Level 1: 10 (max d10) + 3 (CON) = **13 HP**
- Level 2: +6 (avg d10) + 3 (CON) = **22 HP total** (+9)
- Level 3: +6 + 3 = **31 HP total** (+9)
- Level 4: +6 + 3 = **40 HP total** (+9)

#### 2. Proficiency Bonus Increase

Proficiency bonus increases at specific levels:

| Level Range | Proficiency Bonus |
|-------------|-------------------|
| 1-4         | +2                |
| 5-8         | +3                |
| 9-12        | +4                |
| 13-16       | +5                |
| 17-20       | +6                |

**What Uses Proficiency:**
- Attack rolls with proficient weapons
- Spell attack rolls
- Saving throw DCs
- Skill checks (with proficient skills)

**Impact Example:**
- Level 4 Fighter: +2 proficiency → Attack bonus: +2 (prof) + 3 (STR) = +5
- Level 5 Fighter: +3 proficiency → Attack bonus: +3 (prof) + 3 (STR) = +6
- **Result**: +1 to hit, increasing accuracy by ~5%

#### 3. Spell Slot Increases (Casters Only)

Full casters (Wizard, Sorcerer, Cleric, Druid, Warlock) gain spell slots as they level:

| Level | 1st-level | 2nd-level | 3rd-level | 4th-level | 5th-level |
|-------|-----------|-----------|-----------|-----------|-----------|
| 1     | 2         | —         | —         | —         | —         |
| 2     | 3         | —         | —         | —         | —         |
| 3     | 4         | 2         | —         | —         | —         |
| 4     | 4         | 3         | —         | —         | —         |
| 5     | 4         | 3         | 2         | —         | —         |

**Spell Slot Example:**
- **Level 1 Wizard**: 2 × 1st-level spell slots
  - Can cast Magic Missile twice per day
  - Fire Bolt (cantrip) unlimited

- **Level 3 Wizard**: 4 × 1st-level + 2 × 2nd-level spell slots
  - Can cast 6 leveled spells per day total
  - Access to 2nd-level spells (Scorching Ray, Mirror Image, etc.)

**Non-Casters:** Fighter, Rogue, Barbarian gain no spell slots.

#### 4. Class Features (Future Expansion)

In a more advanced tutorial, leveling would also grant:
- **Level 2**: Class-specific features (Fighter: Action Surge, Wizard: School specialization)
- **Level 3**: Subclass choice (Fighter: Champion/Battlemaster, Wizard: School of Evocation)
- **Level 4, 8, 12, 16, 19**: Ability Score Improvements
- **Level 5**: Extra Attack (martial classes), 3rd-level spells (casters)

Tutorial 5 focuses on the core progression mechanics (XP, HP, proficiency, spell slots) and leaves class features for future expansion.

## Progression Strategy

### Optimal XP Farming

**Low-Level Characters (Level 1-2):**
- Fight CR 1/4 monsters (Goblins, Skeletons): 50 XP each
- 6 victories to reach level 2 (300 XP)
- 12 more victories to reach level 3 (600 more XP needed)

**Mid-Level Characters (Level 3-4):**
- Fight CR 1/2 monsters (Orcs): 100 XP each
- 18 victories to reach level 4 from level 3 (1,800 XP needed)

**Progression Curve:**
Early levels come fast (level 1→2 in 6 fights), but slow down dramatically (level 3→4 needs 18 fights).

### Risk vs. Reward

**Higher CR = More Risk, More Reward:**
- CR 1/4 Goblin: 50 XP, easier fight
- CR 1/2 Orc: 100 XP (2x), harder fight
- CR 1 Dire Wolf: 200 XP (4x), much harder

**Recommendation:** Fight monsters slightly below your level for consistent wins and steady XP accumulation.

## Progression Commands Reference

### Award XP Manually

```bash
python3 scripts/progression.py award CHARACTER_NAME XP_AMOUNT
```

**Example:**
```bash
python3 scripts/progression.py award Bob 50
# ✓ Awarded 50 XP to Bob
#   Total XP: 0 → 50
#   XP to level 2: 250 more needed
```

### Award XP via Combat (Automatic)

```bash
python3 scripts/combat.py end CHARACTER_NAME victory --monster-cr CR_VALUE
```

**Example:**
```bash
python3 scripts/combat.py end Bob victory --monster-cr 0.25
# ✓ Victory! Bob has been fully healed.
# ✓ Awarded 50 XP to Bob
#   Total XP: 250 → 300
#
# 🎉 LEVEL UP! Bob is now level 2!
#   Levels gained: 1
#   HP increased: 11 → 20 (+9)
#   Proficiency bonus: +2
#   XP to next level: 600
```

### View XP Table

```bash
python3 scripts/progression.py xp-table
```

Shows all level thresholds and XP requirements.

### View CR XP Awards

```bash
python3 scripts/progression.py cr-xp
```

Shows XP awarded for each Challenge Rating.

## Skill Concepts Demonstrated

### 1. Threshold Triggers

**Concept:** Events that fire automatically when a value crosses a threshold.

**Implementation:**
```python
while new_xp >= XP_THRESHOLDS[new_level]:
    new_level += 1
    levels_gained += 1
```

When XP crosses 300, level-up triggers automatically. No manual intervention needed.

**Real-World Applications:**
- Alert systems (temperature > 100°F → trigger alarm)
- Subscription tiers (usage > 1000 requests → upgrade tier)
- Achievement systems (score > 10000 → unlock badge)

### 2. Progressive Complexity

**Concept:** Features unlock gradually as a system advances, not all at once.

**Implementation in Tutorial 5:**
- Level 1: Basic combat
- Level 2: +HP, better attack bonus
- Level 3: +HP, **2nd-level spell slots** (new capability!)
- Level 4: +HP, +proficiency bonus increases at level 5
- Level 5: +HP, **3rd-level spell slots**, +proficiency (major power spike!)

Each level adds incremental improvements, with occasional big jumps.

**Real-World Applications:**
- SaaS product tiers (Basic → Pro → Enterprise with progressive features)
- Game progression (unlock abilities gradually, not all at start)
- Learning systems (tutorials unlock advanced features after mastery)

### 3. Calculated Growth

**Concept:** Growth follows formulas, not arbitrary numbers.

**Implementation:**
- HP = f(class, level, CON modifier)
- Proficiency = 2 + floor((level - 1) / 4)
- Spell slots = lookup_table(class, level)

Every increase is deterministic and predictable based on rules.

**Real-World Applications:**
- Compound interest calculations
- Performance metrics (response time based on server load formula)
- Pricing tiers (cost = base + (users × per_user_rate))

## Technical Implementation Notes

### Database Schema Changes

Tutorial 5 extends the `characters` table with:

```sql
xp INTEGER DEFAULT 0
```

This single field enables the entire progression system. XP accumulates over time and triggers cascading updates (level, HP, spell slots) when thresholds are crossed.

### Automatic Level-Up Flow

```
1. Combat ends (victory)
2. combat.py calls progression.py award
3. progression.py:
   - Adds XP to character
   - Checks thresholds in while loop
   - If threshold crossed:
     - Increment level
     - Recalculate HP (add hit die avg + CON)
     - Recalculate spell slots
     - Update database
     - Display level-up message
```

### Idempotency Consideration

The level-up system is **not idempotent** - calling `award 300` twice will award 600 XP total. This is intentional: each XP award represents a separate combat victory.

If idempotency were needed (e.g., "set XP to value"), the system would need:
```python
def set_xp(character, new_xp):
    # Set absolute value instead of incrementing
    cursor.execute("UPDATE characters SET xp = ? WHERE name = ?", (new_xp, character))
```

### Why XP in Database, Not Calculated?

XP could theoretically be calculated from combat history, but storing it directly:
- **Simpler queries**: SELECT xp is faster than SUM(combat_xp)
- **Flexible**: Allows manual XP awards (quest rewards, DM fiat)
- **Efficient**: No need to scan combat history on every level check

## Future Enhancements

### Ability Score Improvements (ASI)

D&D 5e grants ASI at levels 4, 8, 12, 16, 19. Characters can:
- Increase two ability scores by 1 each (max 20)
- Increase one ability score by 2 (max 20)
- Take a feat instead

**Implementation:**
- Add ASI tracking to database
- Prompt user at level-up to choose improvements
- Recalculate modifiers, AC, HP, spell DCs

### Class Features

- Level 2: Class-specific features
- Level 3: Subclass choice
- Level 5: Extra Attack or spell level unlocks
- Level 11: Major feature upgrades

### Multiclassing

Allow characters to take levels in multiple classes:
- Track levels per class
- Complex spell slot calculations
- Proficiency and feature restrictions

### Milestone Leveling (Alternative)

Instead of XP, level up after completing story milestones:
- Track quest completion
- Level up at DM discretion
- Simpler but less granular
