#!/usr/bin/env python3
# ABOUTME: Manages character equipment and inventory for combat
# ABOUTME: Handles weapon, armor, and AC calculations

import sqlite3
import json
import argparse
import sys
from pathlib import Path

DB_PATH = Path.home() / ".claude" / "data" / "dnd-dm.db"

# Starting equipment by class (5e PHB)
STARTING_EQUIPMENT = {
    "fighter": {
        "armor": {"name": "Chain Mail", "ac": 16, "dex_bonus": False},
        "shield": {"name": "Shield", "ac_bonus": 2},
        "weapon": {"name": "Longsword", "damage": "1d8", "ability": "str"}
    },
    "wizard": {
        "armor": None,
        "shield": None,
        "weapon": {"name": "Quarterstaff", "damage": "1d6", "ability": "str"}
    },
    "rogue": {
        "armor": {"name": "Leather Armor", "ac": 11, "dex_bonus": True},
        "shield": None,
        "weapon": {"name": "Shortsword", "damage": "1d6", "ability": "dex"}
    },
    "cleric": {
        "armor": {"name": "Chain Mail", "ac": 16, "dex_bonus": False},
        "shield": {"name": "Shield", "ac_bonus": 2},
        "weapon": {"name": "Mace", "damage": "1d6", "ability": "str"}
    }
}

def init_db():
    """Initialize the database and create equipment table if it doesn't exist."""
    DB_PATH.parent.mkdir(parents=True, exist_ok=True)
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS equipment (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            character_name TEXT NOT NULL,
            slot TEXT NOT NULL,
            item_name TEXT NOT NULL,
            stats TEXT NOT NULL,
            FOREIGN KEY (character_name) REFERENCES characters(name) ON DELETE CASCADE
        )
    """)

    # Create unique index on character_name + slot
    cursor.execute("""
        CREATE UNIQUE INDEX IF NOT EXISTS idx_character_slot
        ON equipment(character_name, slot)
    """)

    conn.commit()
    conn.close()

def equip_starting_gear(character_name, character_class):
    """Equip a character with their starting gear based on class."""
    init_db()

    class_lower = character_class.lower()
    if class_lower not in STARTING_EQUIPMENT:
        print(f"✗ Error: Unknown class '{character_class}'", file=sys.stderr)
        return 1

    gear = STARTING_EQUIPMENT[class_lower]
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    try:
        # Check if character exists
        cursor.execute("SELECT name FROM characters WHERE name = ?", (character_name,))
        if not cursor.fetchone():
            print(f"✗ Error: Character '{character_name}' not found", file=sys.stderr)
            return 1

        # Clear existing equipment
        cursor.execute("DELETE FROM equipment WHERE character_name = ?", (character_name,))

        # Equip armor
        if gear["armor"]:
            cursor.execute(
                "INSERT OR REPLACE INTO equipment (character_name, slot, item_name, stats) VALUES (?, ?, ?, ?)",
                (character_name, "armor", gear["armor"]["name"], json.dumps(gear["armor"]))
            )

        # Equip shield
        if gear["shield"]:
            cursor.execute(
                "INSERT OR REPLACE INTO equipment (character_name, slot, item_name, stats) VALUES (?, ?, ?, ?)",
                (character_name, "shield", gear["shield"]["name"], json.dumps(gear["shield"]))
            )

        # Equip weapon
        cursor.execute(
            "INSERT OR REPLACE INTO equipment (character_name, slot, item_name, stats) VALUES (?, ?, ?, ?)",
            (character_name, "weapon", gear["weapon"]["name"], json.dumps(gear["weapon"]))
        )

        conn.commit()
        print(f"✓ Equipped {character_name} with {class_lower} starting gear")
        return 0

    finally:
        conn.close()

def show_equipment(character_name):
    """Display a character's equipped items."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute(
        "SELECT slot, item_name, stats FROM equipment WHERE character_name = ? ORDER BY slot",
        (character_name,)
    )
    results = cursor.fetchall()
    conn.close()

    if not results:
        print(f"No equipment found for {character_name}")
        return 0

    print(f"\n{character_name}'s Equipment:")
    print("=" * 40)
    for slot, item_name, stats_json in results:
        print(f"{slot.capitalize()}: {item_name}")

    return 0

def calculate_ac(character_name):
    """Calculate and display a character's AC based on equipment and DEX."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Get character DEX
    cursor.execute("SELECT dexterity FROM characters WHERE name = ?", (character_name,))
    result = cursor.fetchone()
    if not result:
        print(f"✗ Error: Character '{character_name}' not found", file=sys.stderr)
        conn.close()
        return 1

    dex = result[0]
    dex_mod = (dex - 10) // 2

    # Get equipment
    cursor.execute(
        "SELECT slot, stats FROM equipment WHERE character_name = ?",
        (character_name,)
    )
    equipment = cursor.fetchall()
    conn.close()

    # Calculate AC
    base_ac = 10  # Unarmored
    armor_allows_dex = True
    shield_bonus = 0

    for slot, stats_json in equipment:
        stats = json.loads(stats_json)

        if slot == "armor":
            base_ac = stats["ac"]
            armor_allows_dex = stats.get("dex_bonus", False)
        elif slot == "shield":
            shield_bonus = stats["ac_bonus"]

    # Apply DEX if allowed
    if armor_allows_dex:
        final_ac = base_ac + dex_mod + shield_bonus
    else:
        final_ac = base_ac + shield_bonus

    print(final_ac)
    return 0

def get_weapon_stats(character_name):
    """Get weapon stats for a character (for combat system)."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Get character stats
    cursor.execute(
        "SELECT strength, dexterity, level FROM characters WHERE name = ?",
        (character_name,)
    )
    char_result = cursor.fetchone()
    if not char_result:
        print(f"✗ Error: Character '{character_name}' not found", file=sys.stderr)
        conn.close()
        return 1

    strength, dexterity, level = char_result
    proficiency = 2 + ((level - 1) // 4)

    # Get weapon
    cursor.execute(
        "SELECT stats FROM equipment WHERE character_name = ? AND slot = 'weapon'",
        (character_name,)
    )
    weapon_result = cursor.fetchone()
    conn.close()

    if not weapon_result:
        print(f"✗ Error: No weapon equipped for {character_name}", file=sys.stderr)
        return 1

    weapon = json.loads(weapon_result[0])

    # Calculate attack bonus
    ability_mod = (strength - 10) // 2 if weapon["ability"] == "str" else (dexterity - 10) // 2
    attack_bonus = proficiency + ability_mod

    # Output as JSON
    output = {
        "name": weapon["name"],
        "damage": weapon["damage"],
        "attack_bonus": attack_bonus,
        "ability_modifier": ability_mod
    }
    print(json.dumps(output))
    return 0

def main():
    parser = argparse.ArgumentParser(description="Manage D&D character equipment")
    subparsers = parser.add_subparsers(dest="command", help="Command to execute")

    # Equip command
    equip_parser = subparsers.add_parser("equip", help="Equip starting gear")
    equip_parser.add_argument("character_name", help="Character name")
    equip_parser.add_argument("class_name", help="Character class")

    # Show command
    show_parser = subparsers.add_parser("show", help="Show character equipment")
    show_parser.add_argument("character_name", help="Character name")

    # AC command
    ac_parser = subparsers.add_parser("ac", help="Calculate character AC")
    ac_parser.add_argument("character_name", help="Character name")

    # Weapon command
    weapon_parser = subparsers.add_parser("weapon", help="Get weapon stats for combat")
    weapon_parser.add_argument("character_name", help="Character name")

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        return 1

    if args.command == "equip":
        return equip_starting_gear(args.character_name, args.class_name)
    elif args.command == "show":
        return show_equipment(args.character_name)
    elif args.command == "ac":
        return calculate_ac(args.character_name)
    elif args.command == "weapon":
        return get_weapon_stats(args.character_name)

    return 0

if __name__ == "__main__":
    sys.exit(main())
