#!/usr/bin/env python3
# ABOUTME: D&D character progression and leveling system
# ABOUTME: Handles XP awards, automatic level-ups, stat increases, and progression tracking

import sqlite3
import argparse
import sys
import json
from pathlib import Path

# Database location
DB_PATH = Path.home() / ".claude" / "data" / "dnd-dm.db"

# XP thresholds for levels 1-20 (D&D 5e)
XP_THRESHOLDS = [
    0,      # Level 1
    300,    # Level 2
    900,    # Level 3
    2700,   # Level 4
    6500,   # Level 5
    14000,  # Level 6
    23000,  # Level 7
    34000,  # Level 8
    48000,  # Level 9
    64000,  # Level 10
    85000,  # Level 11
    100000, # Level 12
    120000, # Level 13
    140000, # Level 14
    165000, # Level 15
    195000, # Level 16
    225000, # Level 17
    265000, # Level 18
    305000, # Level 19
    355000  # Level 20
]

# CR to XP mapping for monster defeats
CR_XP_AWARDS = {
    0: 10,
    0.125: 25,
    0.25: 50,
    0.5: 100,
    1: 200,
    2: 450,
    3: 700,
    4: 1100,
    5: 1800
}


def init_db():
    """Initialize database connection."""
    DB_PATH.parent.mkdir(parents=True, exist_ok=True)
    return sqlite3.connect(DB_PATH)


def calculate_modifier(score):
    """Calculate ability modifier from ability score."""
    return (score - 10) // 2


def calculate_hp_increase(character_class, constitution_score):
    """Calculate HP gained on level up."""
    # Hit dice by class
    hit_dice = {
        'fighter': 10,
        'paladin': 10,
        'ranger': 10,
        'barbarian': 12,
        'wizard': 6,
        'sorcerer': 6,
        'rogue': 8,
        'bard': 8,
        'cleric': 8,
        'druid': 8,
        'monk': 8,
        'warlock': 8
    }

    hd = hit_dice.get(character_class.lower(), 8)
    con_mod = calculate_modifier(constitution_score)

    # Average roll + CON modifier
    avg_roll = (hd // 2) + 1
    return avg_roll + con_mod


def calculate_proficiency_bonus(level):
    """Calculate proficiency bonus for a given level."""
    return 2 + ((level - 1) // 4)


def calculate_spell_slots(character_class, level):
    """Calculate spell slots for a character at a given level."""
    full_casters = ['wizard', 'sorcerer', 'cleric', 'druid', 'warlock']

    char_class_lower = character_class.lower()

    if char_class_lower in full_casters:
        # Full caster spell slot progression
        if level == 1:
            return {"1": {"max": 2, "current": 2}}
        elif level == 2:
            return {"1": {"max": 3, "current": 3}}
        elif level == 3:
            return {"1": {"max": 4, "current": 4}, "2": {"max": 2, "current": 2}}
        elif level == 4:
            return {"1": {"max": 4, "current": 4}, "2": {"max": 3, "current": 3}}
        elif level == 5:
            return {"1": {"max": 4, "current": 4}, "2": {"max": 3, "current": 3}, "3": {"max": 2, "current": 2}}
        # Can expand further for higher levels

    # Non-casters or not implemented yet
    return None


def award_xp(character_name, xp_amount):
    """Award XP to a character and handle automatic level-ups."""
    conn = init_db()
    cursor = conn.cursor()

    # Get current character data
    cursor.execute("""
        SELECT level, xp, class, constitution, hp_max, spell_slots
        FROM characters
        WHERE name = ?
    """, (character_name,))

    result = cursor.fetchone()
    if not result:
        print(f"✗ Error: Character '{character_name}' not found", file=sys.stderr)
        conn.close()
        return 1

    (current_level, current_xp, char_class, con_score, hp_max, spell_slots_json) = result

    # Award XP
    new_xp = current_xp + xp_amount
    print(f"✓ Awarded {xp_amount} XP to {character_name}")
    print(f"  Total XP: {current_xp} → {new_xp}")

    # Check for level-ups
    levels_gained = 0
    new_level = current_level

    while new_level < 20 and new_xp >= XP_THRESHOLDS[new_level]:
        new_level += 1
        levels_gained += 1

    if levels_gained > 0:
        # Calculate benefits
        hp_increase = 0
        for _ in range(levels_gained):
            hp_increase += calculate_hp_increase(char_class, con_score)

        new_hp_max = hp_max + hp_increase
        new_prof = calculate_proficiency_bonus(new_level)

        # Calculate new spell slots
        new_spell_slots = calculate_spell_slots(char_class, new_level)
        spell_slots_str = json.dumps(new_spell_slots) if new_spell_slots else None

        # Update character
        cursor.execute("""
            UPDATE characters
            SET level = ?, xp = ?, hp_max = ?, hp_current = ?, spell_slots = ?
            WHERE name = ?
        """, (new_level, new_xp, new_hp_max, new_hp_max, spell_slots_str, character_name))

        conn.commit()

        # Display level-up message
        print(f"\n🎉 LEVEL UP! {character_name} is now level {new_level}!")
        print(f"  Levels gained: {levels_gained}")
        print(f"  HP increased: {hp_max} → {new_hp_max} (+{hp_increase})")
        print(f"  Proficiency bonus: +{new_prof}")

        if new_level < 20:
            xp_to_next = XP_THRESHOLDS[new_level] - new_xp
            print(f"  XP to next level: {xp_to_next}")

        if new_spell_slots and new_spell_slots != (json.loads(spell_slots_json) if spell_slots_json else None):
            print(f"  Spell slots increased!")
            for level_key, slots in sorted(new_spell_slots.items()):
                print(f"    Level {level_key}: {slots['max']} slots")

    else:
        # Just update XP
        cursor.execute("""
            UPDATE characters
            SET xp = ?
            WHERE name = ?
        """, (new_xp, character_name))

        conn.commit()

        # Show progress to next level
        if current_level < 20:
            xp_to_next = XP_THRESHOLDS[current_level] - new_xp
            print(f"  XP to level {current_level + 1}: {xp_to_next} more needed")

    conn.close()
    return 0


def show_xp_table(args):
    """Display XP requirements for each level."""
    print("\nD&D 5e Experience Point Thresholds")
    print("=" * 30)

    for level in range(1, 20):
        xp_current = XP_THRESHOLDS[level - 1]
        xp_next = XP_THRESHOLDS[level]
        xp_needed = xp_next - xp_current

        print(f"Level {level:2d} → {level + 1:2d}: {xp_next:>7,} XP (+{xp_needed:,})")

    print(f"\nLevel 20: MAX LEVEL")
    return 0


def show_cr_xp(args):
    """Display XP awards by CR."""
    print("\nXP Awards by Challenge Rating")
    print("=" * 30)

    for cr, xp in sorted(CR_XP_AWARDS.items()):
        cr_display = f"CR {cr:>5}" if cr >= 1 else f"CR {cr:>5}"
        print(f"{cr_display}: {xp:>5} XP")

    return 0


def main():
    parser = argparse.ArgumentParser(
        description="D&D 5e Character Progression System"
    )
    subparsers = parser.add_subparsers(dest="command", help="Available commands")

    # Award XP command
    award_parser = subparsers.add_parser("award", help="Award XP to a character")
    award_parser.add_argument("character", help="Character name")
    award_parser.add_argument("xp", type=int, help="XP amount to award")

    # Show XP table
    subparsers.add_parser("xp-table", help="Show XP requirements for each level")

    # Show CR XP awards
    subparsers.add_parser("cr-xp", help="Show XP awards by Challenge Rating")

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        return 1

    if args.command == "award":
        return award_xp(args.character, args.xp)
    elif args.command == "xp-table":
        return show_xp_table(args)
    elif args.command == "cr-xp":
        return show_cr_xp(args)


if __name__ == "__main__":
    sys.exit(main())
