# Tutorial 6: Party Combat

Learn to build skills with **collection management**, **priority queues**, and **state synchronization** by implementing a D&D 5e party-based encounter system where multiple heroes face multiple monsters.

## What You'll Learn

### Skill-Building Concepts

1. **Collection Management** - Managing multiple related entities with synchronized state
2. **Priority Queues** - Dynamic ordering by priority (initiative) rather than arrival time
3. **State Synchronization** - Keeping interdependent state consistent across operations

### D&D Features Implemented

- Party creation and roster management
- Multi-combatant encounters (3-5 heroes vs 2-5+ monsters)
- Initiative-based turn order (priority queue)
- Interactive combat: players control heroes, AI controls monsters
- Automatic victory/defeat detection
- XP distribution across party members
- JSON-driven encounter system

## Prerequisites

Complete **Tutorial 5: Character Progression** first. This tutorial extends character and combat systems with party mechanics.

## Installation

Copy the tutorial directory to your Claude skills folder:

```bash
cp -r tutorial-6-party-combat ~/.claude/skills/tutorial-6
```

## Quick Start

### 1. Create Party and Characters

```bash
# Create party
python3 ~/.claude/skills/tutorial-6/scripts/party.py create heroes "The Heroes" --description "Brave adventurers"

# Create characters with party assignment
python3 ~/.claude/skills/tutorial-6/scripts/character.py create "Aria" wizard \
  --str 8 --dex 14 --con 12 --int 16 --wis 13 --cha 10 --party heroes

python3 ~/.claude/skills/tutorial-6/scripts/character.py create "Theron" fighter \
  --str 16 --dex 12 --con 14 --int 10 --wis 13 --cha 8 --party heroes

python3 ~/.claude/skills/tutorial-6/scripts/character.py create "Bob" cleric \
  --str 13 --dex 10 --con 14 --int 10 --wis 16 --cha 12 --party heroes

# View party
python3 ~/.claude/skills/tutorial-6/scripts/party.py show heroes
```

### 2. Seed Databases

```bash
python3 ~/.claude/skills/tutorial-6/scripts/bestiary.py seed
python3 ~/.claude/skills/tutorial-6/scripts/spells.py seed
```

### 3. Prepare Encounter JSON Files

```bash
# Export party to heroes.json
python3 ~/.claude/skills/tutorial-6/scripts/party.py export heroes > heroes.json

# Export monsters to foes.json (auto-numbered: Goblin-1, Goblin-2, Goblin-3)
python3 ~/.claude/skills/tutorial-6/scripts/bestiary.py export Goblin Goblin Goblin > foes.json
```

**Alternative:** Use pre-made encounter files:

```bash
cp ~/.claude/skills/tutorial-6/examples/heroes-level1.json heroes.json
cp ~/.claude/skills/tutorial-6/examples/foes-goblin-ambush.json foes.json
```

### 4. Start Encounter

```bash
python3 ~/.claude/skills/tutorial-6/scripts/encounter.py start heroes.json foes.json
```

**Output:**
```
============================================================
ENCOUNTER STARTED!
============================================================

Initiative Order:
   1. [18] 👤 Theron              (d20: 17, DEX: +1)
   2. [16] 👹 Goblin-1             (d20: 14, DEX: +2)
   3. [14] 👤 Aria                 (d20: 12, DEX: +2)
   4. [12] 👹 Goblin-2             (d20: 10, DEX: +2)
   5. [ 9] 👤 Bob                  (d20: 9, DEX: +0)
   6. [ 8] 👹 Goblin-3             (d20: 6, DEX: +2)

============================================================

Round 1
Turn: Theron (Hero)
HP: 13/13

Available actions:
  encounter.py attack Theron TARGET
  encounter.py show

Targets:
  • Goblin-1 (HP: 7/7, AC: 15)
  • Goblin-2 (HP: 7/7, AC: 15)
  • Goblin-3 (HP: 7/7, AC: 15)
```

### 5. Interactive Combat

**Player Decides Action:**
```bash
# Theron attacks Goblin-1 with longsword
python3 ~/.claude/skills/tutorial-6/scripts/encounter.py attack Theron Goblin-1 longsword
```

**Output:**
```
Theron attacks Goblin-1 with longsword!
  Attack roll: 1d20+5 = 19
  HIT! Damage: 1d8+3 = 9
  Goblin-1: 7 → 0 HP
  💀 Goblin-1 has been DEFEATED!

Goblin-2 attacks Theron!
  Attack roll: 1d20+4 = 12
  MISS! (AC 16)

Goblin-3 attacks Theron!
  Attack roll: 1d20+4 = 18
  HIT! Damage: 1d6+2 = 5
  Theron: 13 → 8 HP

Round 1
Turn: Aria (Hero)
HP: 8/8

Available actions:
  encounter.py attack Aria TARGET
  encounter.py cast Aria SPELL TARGET
  encounter.py show

Targets:
  • Goblin-2 (HP: 7/7, AC: 15)
  • Goblin-3 (HP: 7/7, AC: 15)
```

**Aria Casts Spell:**
```bash
python3 ~/.claude/skills/tutorial-6/scripts/encounter.py cast Aria "Fire Bolt" Goblin-2
```

**Monster Turns:** Monsters act automatically after hero turns—no user input required!

### 6. Victory!

When all foes are defeated, encounter.py automatically ends and outputs results:

```
============================================================
ENCOUNTER ENDED: VICTORY!
============================================================

📊 Combat Summary:
  Rounds: 3
  Total XP: 150
  XP per hero: 50

📄 Results saved to: encounter-results.json

💡 Update characters with:
  progression.py award Aria 50
  progression.py award Theron 50
  progression.py award Bob 50
```

### 7. Award XP

```bash
python3 ~/.claude/skills/tutorial-6/scripts/progression.py award Aria 50
python3 ~/.claude/skills/tutorial-6/scripts/progression.py award Theron 50
python3 ~/.claude/skills/tutorial-6/scripts/progression.py award Bob 50
```

## Tutorial Structure

```
tutorial-6-party-combat/
├── SKILL.md                        # Lean skill file with workflow
├── README.md                       # This file
├── scripts/
│   ├── roll_dice.py               # From Tutorial 1
│   ├── character.py               # Extended with party_id field
│   ├── party.py                   # NEW: Party management
│   ├── bestiary.py                # Extended with export command
│   ├── equipment.py               # From Tutorial 3
│   ├── spells.py                  # From Tutorial 4
│   ├── combat.py                  # From Tutorial 5 (1v1 arena)
│   ├── progression.py             # From Tutorial 5
│   └── encounter.py               # NEW: Interactive party combat
├── references/
│   ├── collection-management.md   # Managing multiple entities
│   ├── priority-queues.md         # Initiative and turn order
│   └── state-synchronization.md   # Keeping state consistent
├── assets/
│   └── data/
│       ├── spells_core.json      # Spell database
│       ├── xp_tables.json        # XP thresholds
│       └── encounter_tables.json # Pre-built encounters
└── examples/
    ├── heroes-level1.json         # Sample party (Aria, Theron, Bob)
    ├── foes-goblin-ambush.json    # 3 Goblins (CR 1/4)
    └── foes-orc-warband.json      # 2 Orcs (CR 1/2)
```

## Core Mechanics

### Initiative (Priority Queue)

At combat start, everyone rolls 1d20 + DEX modifier. Turn order is highest to lowest:

```python
# Roll initiative
d20_roll = roll_dice("1d20")
dex_mod = (abilities['dex'] - 10) // 2
initiative = d20_roll + dex_mod

# Sort by priority (highest first)
combatants.sort(key=lambda c: c['initiative'], reverse=True)
```

**Example Initiative Order:**
```
[18] Theron (Fighter, DEX +1)  ← Acts first
[16] Goblin-1 (DEX +2)
[14] Aria (Wizard, DEX +2)
[12] Goblin-2 (DEX +2)
[ 9] Bob (Cleric, DEX +0)
[ 8] Goblin-3 (DEX +2)          ← Acts last
```

Initiative stays fixed for the entire encounter. When turn_index reaches the end, increment round and restart from index 0.

### Interactive Turns

**Hero Turns (Player Controlled):**
- encounter.py displays available actions
- User decides: attack, cast spell, or show state
- encounter.py executes action, then runs monster turns automatically

**Monster Turns (AI Controlled):**
- encounter.py picks target (hero with lowest HP)
- Rolls attack and damage automatically
- Advances to next turn
- No user input required

### Victory/Defeat Detection

After each action, encounter.py checks:

```python
heroes_alive = [h for h in combatants if h['type'] == 'hero' and not h['is_defeated']]
foes_alive = [f for f in combatants if f['type'] == 'foe' and not f['is_defeated']]

if not foes_alive:
    victory()  # All foes defeated
elif not heroes_alive:
    defeat()   # All heroes defeated
```

Combat ends automatically when one side is fully defeated.

### XP Distribution

XP splits evenly among **all party members**, including defeated ones:

```python
total_xp = sum(foe['cr_xp'] for foe in defeated_foes)
xp_per_hero = total_xp // len(heroes)
```

**Example:**
- 3 Goblins defeated (50 XP each) = 150 total XP
- 3 heroes in party
- Each hero gets 50 XP (150 ÷ 3)

## Key Scripts

### encounter.py (NEW)

The star of Tutorial 6! JSON-driven interactive party combat.

**Start Encounter:**
```bash
encounter.py start HEROES_FILE FOES_FILE
```

Loads heroes.json and foes.json, rolls initiative, begins combat. Creates `encounter-state.json` to track combat.

**Character Attack:**
```bash
encounter.py attack CHARACTER TARGET [WEAPON]
```

Only works on hero turns. Rolls attack (1d20 + STR/DEX), calculates damage, applies to target. Automatically runs monster turns afterward.

**Character Cast Spell:**
```bash
encounter.py cast CHARACTER SPELL TARGET
```

Spell attack or saving throw. Automatically runs monster turns afterward.

**Show State:**
```bash
encounter.py show
```

Display all combatants, HP, turn order, current turn. Use anytime during combat.

### party.py (NEW)

Manage party creation and membership.

**Create Party:**
```bash
party.py create PARTY_ID NAME [--description DESC]
```

**Add Member:**
```bash
party.py add PARTY_ID CHARACTER_NAME [--force]
```

Assigns character to party. Use `--force` to move from another party.

**Remove Member:**
```bash
party.py remove CHARACTER_NAME
```

**Show Party:**
```bash
party.py show PARTY_ID
```

**Export to JSON:**
```bash
party.py export PARTY_ID > heroes.json
```

Exports all party members with stats for encounter.py.

### bestiary.py (Extended)

**Export Monsters:**
```bash
bestiary.py export MONSTER [MONSTER ...] > foes.json
```

Auto-numbers duplicates:
```bash
bestiary.py export Goblin Goblin Goblin > foes.json
# Output: Goblin-1, Goblin-2, Goblin-3
```

Mixed encounter:
```bash
bestiary.py export Goblin Goblin Orc > foes.json
# Output: Goblin-1, Goblin-2, Orc
```

### character.py (Extended)

**Create with Party:**
```bash
character.py create NAME CLASS --str N --dex N ... --party PARTY_ID
```

**List by Party:**
```bash
character.py list --party PARTY_ID
```

## Skill Concepts in Action

### 1. Collection Management

**What it is:** Managing multiple related entities with synchronized state.

**In Tutorial 6:**
```python
# Track 5-10 combatants simultaneously
combatants = [
    {"name": "Theron", "hp_current": 13, "hp_max": 13, "is_defeated": False},
    {"name": "Goblin-1", "hp_current": 7, "hp_max": 7, "is_defeated": False},
    # ... more combatants
]

# Iteration: Display all heroes
for c in combatants:
    if c['type'] == 'hero':
        print(f"{c['name']}: {c['hp_current']}/{c['hp_max']} HP")

# Filtering: Get alive foes
foes_alive = [c for c in combatants if c['type'] == 'foe' and not c['is_defeated']]

# Finding: Locate specific combatant
target = next((c for c in combatants if c['name'] == target_name), None)

# Bulk update: AOE damage hits multiple targets
for foe in foes_in_range:
    foe['hp_current'] = max(0, foe['hp_current'] - damage)
    if foe['hp_current'] <= 0:
        foe['is_defeated'] = True
```

**Real-World Applications:**
- **Multi-user systems**: Chat rooms (track online users, broadcast messages)
- **Inventory management**: Warehouse systems (total stock across locations)
- **Task queues**: Multiple workers processing jobs
- **Fleet management**: Vehicle tracking (find vehicles needing fuel)

**Key Insight:** Collection management requires iteration, filtering, finding, and bulk updates while maintaining state consistency.

### 2. Priority Queues

**What it is:** Ordered processing based on dynamic priorities, not arrival time.

**In Tutorial 6:**
```python
# Roll initiative (priority)
for c in combatants:
    d20_roll = roll_dice("1d20")
    dex_mod = (c['abilities']['dex'] - 10) // 2
    c['initiative'] = d20_roll + dex_mod

# Sort by priority (highest → lowest)
combatants.sort(key=lambda c: c['initiative'], reverse=True)

# Process in priority order
for c in combatants:
    if not c['is_defeated']:
        take_turn(c)
```

**Contrast with FIFO:**
```python
# FIFO (First-In-First-Out): Process in arrival order
queue = ["Alice", "Bob", "Charlie"]  # Alice joined first
# Process: Alice → Bob → Charlie

# Priority Queue: Process by priority
combatants = [
    {"name": "Alice", "initiative": 15},
    {"name": "Bob", "initiative": 22},
    {"name": "Charlie", "initiative": 8}
]
combatants.sort(key=lambda c: c['initiative'], reverse=True)
# Process: Bob (22) → Alice (15) → Charlie (8)
```

**Real-World Applications:**
- **Task scheduling**: CPU schedulers (higher priority processes first)
- **Emergency response**: Triage systems (Priority 1 patients before Priority 3)
- **Customer support**: Premium tier customers served first
- **Network routing**: QoS classes (voice packets before bulk data)

**Key Insight:** Priority queues optimize for importance/urgency rather than chronological order.

### 3. State Synchronization

**What it is:** Keeping multiple interdependent pieces of state consistent across operations.

**In Tutorial 6:**
```python
def apply_damage(state, target_name, damage):
    """Apply damage and synchronize defeated status."""
    target = find_combatant(state, target_name)

    # 1. Update HP
    old_hp = target['hp_current']
    target['hp_current'] = max(0, old_hp - damage)

    # 2. Synchronize defeated status (atomic operation)
    if target['hp_current'] <= 0 and not target['is_defeated']:
        target['is_defeated'] = True
        print(f"💀 {target_name} has been DEFEATED!")

    # 3. Log action
    state['combat_log'].append({
        'actor': attacker,
        'target': target_name,
        'damage': damage
    })

    # 4. Save synchronized state
    save_state(state)

def advance_turn(state):
    """Advance turn and cascade updates."""
    # 1. Increment turn index
    state['turn_index'] += 1

    # 2. Skip defeated combatants
    while state['turn_index'] < len(state['combatants']):
        if not state['combatants'][state['turn_index']]['is_defeated']:
            break
        state['turn_index'] += 1

    # 3. Check for new round
    if state['turn_index'] >= len(state['combatants']):
        state['round'] += 1
        state['turn_index'] = 0

    # 4. Check victory/defeat (synchronized check)
    check_combat_end(state)

    # 5. Save state
    save_state(state)
```

**Cascading Updates Example:**
```
Goblin-1 takes 9 damage
  ↓
HP: 7 → 0 (state update #1)
  ↓
is_defeated: False → True (state update #2)
  ↓
Skip Goblin-1 in turn order (state update #3)
  ↓
Check if all foes defeated (state update #4)
  ↓
Victory condition met → end combat (state update #5)
```

**Real-World Applications:**
- **Distributed systems**: Database replication (all nodes have consistent data)
- **Transaction management**: Bank transfers (debit and credit happen together or not at all)
- **UI state management**: React/Vue (component state synced with DOM)
- **Multiplayer games**: Player positions broadcasted to all clients

**Key Insight:** State synchronization prevents partial updates, stale data, and inconsistent state.

## Combining the Concepts

The magic happens when all three work together:

1. **Collection Management**: Track 6 combatants (3 heroes + 3 goblins)
2. **Priority Queue**: Roll initiative, sort by priority, determine turn order
3. **State Synchronization**:
   - Theron hits Goblin-1 for 9 damage
   - Goblin-1 HP: 7 → 0
   - Goblin-1 defeated status: False → True
   - Check victory (all foes defeated?)
   - Save state
   - Advance to next turn

This creates a **self-managing encounter system** that:
- Manages multiple entities simultaneously
- Processes turns in priority order
- Keeps all state synchronized
- Automatically detects victory/defeat
- Requires minimal manual intervention

## Example Session

Here's a complete party combat encounter:

```bash
# ============================================================
# SETUP
# ============================================================

# Create party
python3 scripts/party.py create heroes "The Heroes"

# Create characters
python3 scripts/character.py create "Aria" wizard \
  --str 8 --dex 14 --con 12 --int 16 --wis 13 --cha 10 --party heroes

python3 scripts/character.py create "Theron" fighter \
  --str 16 --dex 12 --con 14 --int 10 --wis 13 --cha 8 --party heroes

python3 scripts/character.py create "Bob" cleric \
  --str 13 --dex 10 --con 14 --int 10 --wis 16 --cha 12 --party heroes

# Seed databases
python3 scripts/bestiary.py seed
python3 scripts/spells.py seed

# Export to JSON
python3 scripts/party.py export heroes > heroes.json
python3 scripts/bestiary.py export Goblin Goblin Goblin > foes.json

# ============================================================
# COMBAT
# ============================================================

# Start encounter
python3 scripts/encounter.py start heroes.json foes.json

# Output:
# Initiative Order:
#   1. [18] Theron
#   2. [16] Goblin-1
#   3. [14] Aria
#   4. [12] Goblin-2
#   5. [ 9] Bob
#   6. [ 8] Goblin-3
#
# Round 1, Turn: Theron

# Theron attacks Goblin-1
python3 scripts/encounter.py attack Theron Goblin-1 longsword
# Theron attacks Goblin-1 with longsword!
#   Attack roll: 1d20+5 = 19
#   HIT! Damage: 1d8+3 = 9
#   Goblin-1: 7 → 0 HP
#   💀 Goblin-1 has been DEFEATED!
#
# Goblin-2 attacks Theron!
#   Attack roll: 1d20+4 = 12
#   MISS! (AC 16)
#
# Goblin-3 attacks Theron!
#   Attack roll: 1d20+4 = 18
#   HIT! Damage: 1d6+2 = 5
#   Theron: 13 → 8 HP
#
# Round 1, Turn: Aria

# Aria casts Fire Bolt
python3 scripts/encounter.py cast Aria "Fire Bolt" Goblin-2
# Aria casts Fire Bolt on Goblin-2!
#   Attack roll: 1d20+5 = 20
#   HIT! Damage: 1d10 = 7
#   Goblin-2: 7 → 0 HP
#   💀 Goblin-2 has been DEFEATED!
#
# Goblin-3 attacks Theron!
#   Attack roll: 1d20+4 = 8
#   MISS! (AC 16)
#
# Round 1, Turn: Bob

# Bob attacks Goblin-3 with mace
python3 scripts/encounter.py attack Bob Goblin-3 mace
# Bob attacks Goblin-3 with mace!
#   Attack roll: 1d20+3 = 15
#   HIT! Damage: 1d6+1 = 4
#   Goblin-3: 7 → 3 HP
#
# Goblin-3 attacks Aria!  (AI targets lowest HP)
#   Attack roll: 1d20+4 = 19
#   HIT! Damage: 1d6+2 = 5
#   Aria: 8 → 3 HP
#
# Round 2, Turn: Theron

# Theron finishes Goblin-3
python3 scripts/encounter.py attack Theron Goblin-3 longsword
# Theron attacks Goblin-3 with longsword!
#   Attack roll: 1d20+5 = 17
#   HIT! Damage: 1d8+3 = 7
#   Goblin-3: 3 → 0 HP
#   💀 Goblin-3 has been DEFEATED!
#
# ============================================================
# ENCOUNTER ENDED: VICTORY!
# ============================================================
#
# 📊 Combat Summary:
#   Rounds: 2
#   Total XP: 150
#   XP per hero: 50
#
# 📄 Results saved to: encounter-results.json
#
# 💡 Update characters with:
#   progression.py award Aria 50
#   progression.py award Theron 50
#   progression.py award Bob 50

# ============================================================
# POST-COMBAT
# ============================================================

# Award XP
python3 scripts/progression.py award Aria 50
python3 scripts/progression.py award Theron 50
python3 scripts/progression.py award Bob 50

# Check final stats
python3 scripts/character.py show Aria
# Aria (wizard, Level 1)
# XP: 50/300 (need 250 more)
# HP: 8/8
```

## State Files

### encounter-state.json

Active combat state, saved after each turn:

```json
{
  "round": 2,
  "turn_index": 3,
  "combatants": [
    {
      "name": "Theron",
      "type": "hero",
      "hp_current": 8,
      "hp_max": 13,
      "ac": 16,
      "is_defeated": false,
      "initiative": 18
    },
    {
      "name": "Goblin-1",
      "type": "foe",
      "hp_current": 0,
      "hp_max": 7,
      "is_defeated": true,
      "initiative": 16
    }
  ],
  "combat_log": [
    {"actor": "Theron", "action": "attack", "target": "Goblin-1", "damage": 9},
    {"actor": "Goblin-2", "action": "attack", "target": "Theron", "damage": 0},
    {"actor": "Aria", "action": "cast", "spell": "Fire Bolt", "target": "Goblin-2", "damage": 7}
  ]
}
```

### encounter-results.json

Final results, created when combat ends:

```json
{
  "outcome": "victory",
  "rounds": 2,
  "heroes": [
    {"name": "Aria", "hp_final": 3, "hp_max": 8, "xp_earned": 50},
    {"name": "Theron", "hp_final": 8, "hp_max": 13, "xp_earned": 50},
    {"name": "Bob", "hp_final": 10, "hp_max": 10, "xp_earned": 50}
  ],
  "foes": [
    {"name": "Goblin-1", "hp_final": 0, "hp_max": 7},
    {"name": "Goblin-2", "hp_final": 0, "hp_max": 7},
    {"name": "Goblin-3", "hp_final": 0, "hp_max": 7}
  ],
  "total_xp": 150,
  "xp_per_hero": 50
}
```

## Extension Ideas

Once you've completed this tutorial, try extending it:

1. **AOE Spells**
   - Fireball hits all foes in range
   - Apply damage to multiple targets simultaneously
   - Check if multiple foes defeated in single action

2. **Healing Spells During Combat**
   - Cure Wounds restores HP mid-fight
   - Allow targeting allies instead of enemies
   - Track spell slots consumed during encounter

3. **Conditions and Status Effects**
   - Stunned: Skip turns
   - Poisoned: Disadvantage on attacks
   - Prone: Melee attacks have advantage against you

4. **Monster Spellcasting**
   - Add spellcasting monsters (Goblin Shaman, Drow Mage)
   - AI decides when to cast vs attack
   - Track monster spell slots

5. **Terrain and Cover**
   - Half cover: +2 AC
   - Three-quarters cover: +5 AC
   - Difficult terrain: Movement restrictions

6. **Death Saving Throws**
   - Defeated heroes make death saves
   - Three successes: Stabilize at 0 HP
   - Three failures: Permanent death

7. **Encounter Templates**
   - Load pre-built encounters from encounter_tables.json
   - Difficulty ratings: Easy, Medium, Hard, Deadly
   - Auto-scale encounters based on party level

## What's Next?

**Tutorial 7: Advanced Combat Mechanics** will teach:
- **Reactions** - Opportunity attacks, Shield spell, Counterspell
- **Legendary Actions** - Boss monsters with extra actions
- **Monster Spellcasting** - AI decides when to cast vs attack
- **Conditions and Status Effects** - Stunned, paralyzed, poisoned
- **Environmental Hazards** - Traps, lava, collapsing ceilings

You'll build complex boss encounters with mechanics that challenge even experienced players.

## Troubleshooting

### "No active encounter found"

You need to start an encounter first:

```bash
encounter.py start heroes.json foes.json
```

### "Not CHARACTER's turn"

Check turn order with `encounter.py show`. Only the current character can act.

### "Target not found"

Verify target name exactly matches (case-sensitive). Use `encounter.py show` to see combatant names.

### "Monster already defeated"

Cannot target defeated foes. Choose a different target.

### "Character not in party"

Add character to party first:

```bash
party.py add PARTY_ID CHARACTER_NAME
```

Or create character with `--party` flag:

```bash
character.py create NAME CLASS --party PARTY_ID ...
```

### JSON Parse Error

Validate JSON format:

```bash
# Check if valid JSON
python3 -m json.tool heroes.json
python3 -m json.tool foes.json
```

Ensure files match expected structure (see examples/ directory).

## Files Overview

### Core Scripts

- **`encounter.py`** (NEW) - JSON-driven interactive party combat, turn management, AI monster targeting
- **`party.py`** (NEW) - Party creation, membership, export to JSON
- **`bestiary.py`** - Extended with export command for foes.json
- **`character.py`** - Extended with party_id field

### From Previous Tutorials

- **`roll_dice.py`** - Dice rolling (Tutorial 1)
- **`equipment.py`** - Equipment and AC (Tutorial 3)
- **`spells.py`** - Spell database (Tutorial 4)
- **`combat.py`** - 1v1 arena combat (Tutorial 5)
- **`progression.py`** - XP awards and leveling (Tutorial 5)

### Data Files

- **`assets/data/encounter_tables.json`** - Pre-built encounter templates
- **`assets/data/spells_core.json`** - Spell database
- **`assets/data/xp_tables.json`** - XP thresholds and CR awards

### Documentation

- **`references/collection-management.md`** - Managing multiple entities, iteration, filtering, bulk updates
- **`references/priority-queues.md`** - Initiative system, dynamic priorities, turn order
- **`references/state-synchronization.md`** - Keeping state consistent, atomic operations, cascading updates

### Example Files

- **`examples/heroes-level1.json`** - Sample party (Aria, Theron, Bob)
- **`examples/foes-goblin-ambush.json`** - 3 Goblins (CR 1/4 each)
- **`examples/foes-orc-warband.json`** - 2 Orcs (CR 1/2 each)

## Summary

Tutorial 6 teaches you to build **multi-entity management systems** using:

1. **Collection Management** - Track multiple entities (heroes + monsters) with synchronized state
2. **Priority Queues** - Initiative-based turn order (dynamic priority, not arrival time)
3. **State Synchronization** - Keep HP, defeated status, turn order, and victory conditions consistent

These concepts apply far beyond D&D—use them to build:
- Multi-user chat systems
- Inventory management across locations
- Task scheduling with priorities
- Emergency triage systems
- Distributed systems with synchronized state
- Multiplayer game state management

Now go forth and build epic party encounters! ⚔️
