# Collection Management

## Overview

**Collection Management** is the practice of maintaining multiple related entities with synchronized state. Tutorial 6 demonstrates this through party combat, where you track 5-10 combatants simultaneously, each with their own HP, AC, position in initiative, and defeated status.

## Core Concept

Instead of managing a single character vs a single monster, you now manage:
- **Heroes collection**: 3-5 player characters
- **Foes collection**: 2-5+ monsters
- **Combatants collection**: All heroes + foes in initiative order

Each entity must maintain its own state while staying synchronized with the overall combat state.

## Implementation in Tutorial 6

### Data Structure

```python
# encounter-state.json
{
  "round": 1,
  "turn_index": 0,
  "combatants": [
    {
      "name": "Theron",
      "type": "hero",
      "hp_current": 13,
      "hp_max": 13,
      "ac": 16,
      "is_defeated": false,
      "initiative": 18
    },
    {
      "name": "Goblin-1",
      "type": "foe",
      "hp_current": 7,
      "hp_max": 7,
      "ac": 15,
      "is_defeated": false,
      "initiative": 16
    },
    // ... more combatants
  ]
}
```

### Key Operations

**1. Iteration**
```python
# Display all heroes
for c in state['combatants']:
    if c['type'] == 'hero':
        print(f"{c['name']}: {c['hp_current']}/{c['hp_max']} HP")
```

**2. Filtering**
```python
# Get all alive foes
foes_alive = [c for c in state['combatants']
              if c['type'] == 'foe' and not c['is_defeated']]
```

**3. Finding Specific Entity**
```python
# Find combatant by name
target = None
for c in state['combatants']:
    if c['name'] == target_name:
        target = c
        break
```

**4. Bulk Updates**
```python
# AOE damage: hit all foes in range
for foe in foes_in_range:
    foe['hp_current'] = max(0, foe['hp_current'] - damage)
    if foe['hp_current'] <= 0:
        foe['is_defeated'] = True
```

**5. Victory/Defeat Conditions**
```python
# Check if combat should end
heroes_alive = [c for c in combatants if c['type'] == 'hero' and not c['is_defeated']]
foes_alive = [c for c in combatants if c['type'] == 'foe' and not c['is_defeated']]

if not foes_alive:
    victory()
elif not heroes_alive:
    defeat()
```

## Challenges

### 1. State Consistency

**Problem:** Updating one entity without updating related state.

**Example:**
```python
# ❌ Wrong: HP reduced but is_defeated not updated
target['hp_current'] = 0
# Combatant still acts on their turn!

# ✅ Correct: Update both HP and defeated status
target['hp_current'] = 0
if target['hp_current'] <= 0:
    target['is_defeated'] = True
```

### 2. Collection Modification During Iteration

**Problem:** Removing items while iterating breaks the loop.

**Example:**
```python
# ❌ Wrong: Modifying list during iteration
for combatant in combatants:
    if combatant['is_defeated']:
        combatants.remove(combatant)  # Breaks iteration!

# ✅ Correct: Filter to new list
combatants = [c for c in combatants if not c['is_defeated']]
```

### 3. Index Management

**Problem:** turn_index becomes invalid after entities are defeated.

**Solution:** Skip defeated combatants when advancing:
```python
def advance_turn(state):
    state['turn_index'] += 1

    # Skip defeated combatants
    while state['turn_index'] < len(state['combatants']):
        if not state['combatants'][state['turn_index']]['is_defeated']:
            break
        state['turn_index'] += 1
```

## Real-World Applications

### 1. Multi-User Systems

**Chat Rooms:**
```python
# Track multiple users
users = [
    {"name": "Alice", "status": "online", "last_message": "2025-01-13 10:30"},
    {"name": "Bob", "status": "away", "last_message": "2025-01-13 10:15"},
    # ...
]

# Broadcast message to all online users
for user in users:
    if user['status'] == 'online':
        send_message(user, message)
```

### 2. Inventory Management

**Warehouse System:**
```python
# Track items across multiple locations
inventory = [
    {"sku": "ABC123", "location": "A1", "quantity": 50},
    {"sku": "ABC123", "location": "B2", "quantity": 30},
    {"sku": "DEF456", "location": "A1", "quantity": 20},
]

# Total stock for SKU across all locations
def total_stock(sku):
    return sum(item['quantity'] for item in inventory if item['sku'] == sku)
```

### 3. Task Queues

**Job Processing:**
```python
# Track multiple worker processes
workers = [
    {"id": 1, "status": "busy", "current_task": "task_123"},
    {"id": 2, "status": "idle", "current_task": None},
    {"id": 3, "status": "busy", "current_task": "task_124"},
]

# Assign task to next available worker
idle_workers = [w for w in workers if w['status'] == 'idle']
if idle_workers:
    worker = idle_workers[0]
    worker['status'] = 'busy'
    worker['current_task'] = new_task_id
```

### 4. Fleet Management

**Vehicle Tracking:**
```python
# Track delivery vehicles
fleet = [
    {"vehicle_id": "V001", "location": (lat, lng), "fuel": 75, "on_route": True},
    {"vehicle_id": "V002", "location": (lat, lng), "fuel": 30, "on_route": False},
]

# Find vehicles needing fuel
low_fuel = [v for v in fleet if v['fuel'] < 25]
```

## Best Practices

### 1. Immutable IDs

Use stable identifiers (names, IDs) rather than list indices:

```python
# ✅ Good: Find by name
def find_combatant(name):
    for c in combatants:
        if c['name'] == name:
            return c
    return None

# ❌ Bad: Use index (breaks when list changes)
target = combatants[5]  # What if combatant 4 was removed?
```

### 2. Separate Collections for Different Types

```python
# Instead of one big list with type filtering:
all_entities = [heroes..., monsters..., npcs..., items...]

# Use separate collections:
heroes = [...]
monsters = [...]
npcs = [...]
items = [...]
```

### 3. Defensive Copying

```python
# When passing collections to functions, consider if mutation is intended
def process_heroes(heroes):
    # If you don't want to modify original:
    local_heroes = heroes.copy()
    # Work with local_heroes
```

### 4. Validation on Collection Operations

```python
def apply_damage(combatants, target_name, damage):
    target = find_combatant(combatants, target_name)

    if not target:
        raise ValueError(f"Target {target_name} not found")

    if target['is_defeated']:
        raise ValueError(f"{target_name} is already defeated")

    # Apply damage
    target['hp_current'] = max(0, target['hp_current'] - damage)
```

## Summary

Collection management is essential when building systems that track multiple related entities. Key skills:

1. **Iteration** - Loop through collections efficiently
2. **Filtering** - Extract subsets based on conditions
3. **Finding** - Locate specific entities by ID/name
4. **Bulk updates** - Apply changes to multiple entities
5. **State consistency** - Keep all related state synchronized

Master these patterns in Tutorial 6's combat system, and you'll be ready to build multi-user systems, inventory trackers, task managers, and more.
