# Priority Queues

## Overview

A **priority queue** is a data structure where elements are processed based on their priority, not their arrival order. Tutorial 6's initiative system is a perfect example: combatants act based on their initiative roll (priority), not the order they joined combat.

## FIFO Queue vs Priority Queue

**FIFO (First-In-First-Out):**
```python
# Regular queue: process in arrival order
queue = ["Alice", "Bob", "Charlie"]
# Process: Alice → Bob → Charlie
```

**Priority Queue:**
```python
# Initiative rolls determine order
combatants = [
    {"name": "Alice", "initiative": 15},
    {"name": "Bob", "initiative": 22},
    {"name": "Charlie", "initiative": 8}
]

# Sort by priority (initiative)
combatants.sort(key=lambda c: c['initiative'], reverse=True)
# Process: Bob (22) → Alice (15) → Charlie (8)
```

## Initiative System (Tutorial 6)

### Rolling Initiative

At combat start, everyone rolls 1d20 + DEX modifier:

```python
def roll_initiative(combatants):
    for combatant in combatants:
        d20_roll = roll_dice("1d20")
        dex_mod = calculate_modifier(combatant['abilities']['dex'])
        combatant['initiative'] = d20_roll + dex_mod

    # Sort highest to lowest (priority queue!)
    combatants.sort(key=lambda c: c['initiative'], reverse=True)
    return combatants
```

### Turn Order

```python
# Example after rolling:
# [18] Theron (Fighter)
# [16] Goblin-1
# [14] Aria (Wizard)
# [12] Goblin-2
# [9] Bob (Cleric)

# Round 1:
#   Turn 1: Theron acts
#   Turn 2: Goblin-1 acts
#   Turn 3: Aria acts
#   Turn 4: Goblin-2 acts
#   Turn 5: Bob acts
# Round 2: Repeat in same order
```

### Handling Ties

```python
# If two combatants have same initiative, sort by DEX
combatants.sort(
    key=lambda c: (c['initiative'], c['abilities']['dex']),
    reverse=True
)
```

## Real-World Applications

### 1. Task Scheduling

**CPU Scheduler:**
```python
tasks = [
    {"name": "Task A", "priority": 1, "duration": 10},  # High priority
    {"name": "Task B", "priority": 3, "duration": 5},   # Low priority
    {"name": "Task C", "priority": 2, "duration": 15},  # Medium priority
]

# Process by priority (lowest number = highest priority)
tasks.sort(key=lambda t: t['priority'])
# Order: Task A → Task C → Task B
```

### 2. Emergency Response

**Triage System:**
```python
patients = [
    {"name": "Patient A", "severity": "minor"},    # Priority 3
    {"name": "Patient B", "severity": "critical"}, # Priority 1
    {"name": "Patient C", "severity": "urgent"},   # Priority 2
]

severity_priority = {"critical": 1, "urgent": 2, "minor": 3}
patients.sort(key=lambda p: severity_priority[p['severity']])
# Order: Patient B → Patient C → Patient A
```

### 3. Customer Support Tickets

```python
tickets = [
    {"id": 101, "tier": "premium", "wait_time": 10},
    {"id": 102, "tier": "basic", "wait_time": 120},
    {"id": 103, "tier": "premium", "wait_time": 5},
]

tier_priority = {"premium": 1, "basic": 2}

# Sort by tier first, then wait time
tickets.sort(key=lambda t: (tier_priority[t['tier']], -t['wait_time']))
# Premium customers with longest wait first
```

### 4. Network Packet Routing

```python
packets = [
    {"id": 1, "qos_class": "video", "timestamp": 100},
    {"id": 2, "qos_class": "bulk", "timestamp": 50},
    {"id": 3, "qos_class": "voice", "timestamp": 75},
]

qos_priority = {"voice": 1, "video": 2, "bulk": 3}
packets.sort(key=lambda p: qos_priority[p['qos_class']])
# Voice → Video → Bulk
```

## Dynamic Priority Queues

Unlike static queues, priority can change:

```python
# Initiative can be rerolled mid-combat
def reroll_initiative(combatant):
    d20_roll = roll_dice("1d20")
    dex_mod = calculate_modifier(combatant['abilities']['dex'])
    combatant['initiative'] = d20_roll + dex_mod

    # Re-sort entire queue
    all_combatants.sort(key=lambda c: c['initiative'], reverse=True)
```

## Implementation Patterns

### 1. Simple Sort (Tutorial 6 Approach)

```python
# Store all items in list, sort by priority
items = [...]
items.sort(key=lambda x: x['priority'], reverse=True)

# Process in order
for item in items:
    process(item)
```

**Pros:** Simple, easy to understand
**Cons:** Must re-sort if priorities change

### 2. Python heapq Module

```python
import heapq

# Min-heap (lowest value = highest priority)
heap = []
heapq.heappush(heap, (priority, item))
highest_priority_item = heapq.heappop(heap)
```

### 3. Custom Priority Queue Class

```python
class PriorityQueue:
    def __init__(self):
        self.items = []

    def push(self, item, priority):
        self.items.append((priority, item))
        self.items.sort(key=lambda x: x[0], reverse=True)

    def pop(self):
        return self.items.pop(0)[1] if self.items else None
```

## Summary

Priority queues process items based on importance/priority rather than arrival order. Tutorial 6's initiative system teaches:

1. **Sorting by priority** - Initiative rolls create turn order
2. **Dynamic priorities** - Initiative can be rerolled
3. **Tie-breaking** - Use secondary sort keys (DEX)
4. **Skipping items** - Defeated combatants are skipped

This pattern applies to task scheduling, emergency response, customer support, network routing, and any system where some items are more urgent than others.
