# State Synchronization

## Overview

**State Synchronization** ensures that multiple interdependent pieces of state remain consistent when operations affect more than one entity. In Tutorial 6, this means keeping HP, defeated status, turn order, and combat outcome all synchronized across multiple combatants.

## The Challenge

When one entity's state changes, other state may need to update:

```python
# Simple: Apply damage
goblin['hp_current'] -= 8

# But also need to:
# - Mark as defeated if HP ≤ 0
# - Remove from initiative if defeated
# - Check if combat should end (all foes defeated?)
# - Update combat log
```

## Synchronization Patterns in Tutorial 6

### 1. Atomic Operations

Group related state changes together:

```python
def apply_damage(state, target_name, damage):
    # Find target
    target = find_combatant(state, target_name)

    # Update HP
    old_hp = target['hp_current']
    target['hp_current'] = max(0, old_hp - damage)

    # Synchronize defeated status
    if target['hp_current'] <= 0 and not target['is_defeated']:
        target['is_defeated'] = True
        print(f"💀 {target_name} has been DEFEATED!")

    # All changes happen together (atomic)
```

### 2. Cascading Updates

One change triggers a chain of related updates:

```python
def advance_turn(state):
    # 1. Increment turn index
    state['turn_index'] += 1

    # 2. Skip defeated combatants
    while state['turn_index'] < len(state['combatants']):
        if not state['combatants'][state['turn_index']]['is_defeated']:
            break
        state['turn_index'] += 1

    # 3. Check for new round
    if state['turn_index'] >= len(state['combatants']):
        state['round'] += 1
        state['turn_index'] = 0

    # 4. Check victory/defeat
    check_combat_end(state)  # May trigger end_encounter()

    # 5. Save state
    save_state(state)
```

### 3. Victory/Defeat Conditions

Multiple entity states determine overall outcome:

```python
def check_combat_end(state):
    heroes_alive = [c for c in state['combatants']
                    if c['type'] == 'hero' and not c['is_defeated']]

    foes_alive = [c for c in state['combatants']
                  if c['type'] == 'foe' and not c['is_defeated']]

    # Synchronized check: ALL foes defeated → victory
    if not foes_alive:
        end_encounter(state, 'victory')
        return True

    # Synchronized check: ALL heroes defeated → defeat
    elif not heroes_alive:
        end_encounter(state, 'defeat')
        return True

    return False
```

### 4. AOE Damage (Mass Update)

Single action affects multiple entities:

```python
def cast_fireball(state, caster, targets, damage):
    """Fireball hits multiple targets simultaneously."""

    for target in targets:
        # Each target takes damage
        apply_damage(state, target['name'], damage)

        # Log each hit
        state['combat_log'].append({
            'actor': caster['name'],
            'action': 'fireball',
            'target': target['name'],
            'damage': damage
        })

    # Save synchronized state once at end
    save_state(state)
```

## Common Synchronization Bugs

### 1. Partial Updates

**Problem:** Update one field but forget related field.

```python
# ❌ Bug: HP set to 0 but is_defeated not updated
goblin['hp_current'] = 0
# Goblin still acts on their turn!

# ✅ Fix: Always update both
goblin['hp_current'] = 0
if goblin['hp_current'] <= 0:
    goblin['is_defeated'] = True
```

### 2. Race Conditions

**Problem:** Multiple updates conflict.

```python
# ❌ Bug: Two processes update simultaneously
# Process A: goblin['hp'] = 5
# Process B: goblin['hp'] = 3
# Final state: unpredictable!

# ✅ Fix: Single source of truth, sequential updates
# Use locks, transactions, or single-threaded processing
```

### 3. Stale Data

**Problem:** Using outdated cached values.

```python
# ❌ Bug: Check combat end at start of turn, but foe defeated mid-turn
foes_alive = get_foes_alive()  # Cached at turn start
# ... combat happens ...
# Foe defeated but foes_alive still has stale data

# ✅ Fix: Check freshly each time
def check_combat_end(state):
    foes_alive = [c for c in state['combatants']
                  if c['type'] == 'foe' and not c['is_defeated']]  # Fresh check
```

## Real-World Applications

### 1. Distributed Systems

**Database Replication:**
```python
# Write to primary database
primary_db.update(user_id=123, balance=500)

# Synchronize to replicas
for replica in replicas:
    replica.update(user_id=123, balance=500)

# All nodes have consistent state
```

### 2. Transaction Management

**Bank Transfer:**
```python
def transfer_money(from_account, to_account, amount):
    # All-or-nothing: both updates succeed or both fail

    from_account.balance -= amount
    to_account.balance += amount

    # If any step fails, rollback both
    if from_account.balance < 0:
        raise InsufficientFunds()
        # Rollback: restore both balances
```

### 3. UI State Management

**React Component State:**
```python
# Shopping cart state synchronized with UI
def add_to_cart(item):
    # 1. Update cart state
    cart.items.append(item)

    # 2. Update total price (synchronized)
    cart.total = sum(i.price for i in cart.items)

    # 3. Update item count badge (synchronized)
    cart.count = len(cart.items)

    # 4. Re-render UI (synchronized with state)
    render_cart(cart)
```

### 4. Multiplayer Games

**Player Position Sync:**
```python
# Player moves on their client
player1.position = (x, y)

# Broadcast to all other clients
for client in other_clients:
    client.update_player_position(player1.id, (x, y))

# All clients show consistent game state
```

## Best Practices

### 1. Single Source of Truth

```python
# ✅ Good: State lives in one place
state = {
    'combatants': [...]  # HP, defeated status stored here
}

# ❌ Bad: Duplicate state in multiple places
heroes_hp = {...}
heroes_defeated = {...}
combatants = {...}
# Now you have to keep 3 structures in sync!
```

### 2. Transaction-Like Updates

```python
def multi_entity_update(state, updates):
    """Apply all updates or none (atomic)."""

    # Validate all updates first
    for update in updates:
        if not is_valid(update):
            raise ValueError("Invalid update")

    # All valid: apply all
    for update in updates:
        apply_update(state, update)

    save_state(state)
```

### 3. Event-Driven Updates

```python
# When one state changes, trigger related updates

def on_combatant_defeated(combatant):
    combatant['is_defeated'] = True

    # Trigger cascading updates
    update_initiative_order()
    check_combat_end()
    log_defeat_event(combatant)
```

### 4. Consistency Checks

```python
def validate_state(state):
    """Assert state is consistent."""

    # Check: defeated combatants have HP ≤ 0
    for c in state['combatants']:
        if c['is_defeated'] and c['hp_current'] > 0:
            raise InconsistentState(f"{c['name']} defeated but has {c['hp_current']} HP")

    # Check: turn_index within bounds
    if state['turn_index'] >= len(state['combatants']):
        raise InconsistentState("turn_index out of range")
```

## Summary

State synchronization keeps multiple interdependent pieces of state consistent. Tutorial 6 teaches:

1. **Atomic operations** - Group related updates
2. **Cascading updates** - One change triggers related changes
3. **Victory conditions** - Multiple entity states determine outcome
4. **Mass updates** - AOE affects multiple entities simultaneously

Apply these patterns to distributed systems, transactions, UI state management, multiplayer games, and any system where multiple pieces of state must stay synchronized.
