#!/usr/bin/env python3
# ABOUTME: Manages the bestiary database for D&D monsters
# ABOUTME: Handles CRUD operations for monsters with CR-based selection

import sqlite3
import json
import argparse
import sys
from pathlib import Path

DB_PATH = Path.home() / ".claude" / "data" / "dnd-dm.db"

def init_db():
    """Initialize the database and create bestiary table if it doesn't exist."""
    DB_PATH.parent.mkdir(parents=True, exist_ok=True)
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS bestiary (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL UNIQUE,
            cr REAL NOT NULL,
            stat_block TEXT NOT NULL
        )
    """)

    conn.commit()
    conn.close()

def add_monster(name, cr, stat_block):
    """Add a monster to the bestiary."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    try:
        # Validate JSON
        json.loads(stat_block)

        cursor.execute(
            "INSERT INTO bestiary (name, cr, stat_block) VALUES (?, ?, ?)",
            (name, cr, stat_block)
        )
        conn.commit()
        print(f"✓ Added {name} (CR {cr}) to bestiary")
        return 0
    except json.JSONDecodeError:
        print(f"✗ Error: Invalid JSON in stat_block", file=sys.stderr)
        return 1
    except sqlite3.IntegrityError:
        print(f"✗ Error: Monster '{name}' already exists", file=sys.stderr)
        return 1
    finally:
        conn.close()

def show_monster(name):
    """Display a monster's full stat block."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("SELECT name, cr, stat_block FROM bestiary WHERE name = ?", (name,))
    result = cursor.fetchone()
    conn.close()

    if not result:
        print(f"✗ Error: Monster '{name}' not found", file=sys.stderr)
        return 1

    name, cr, stat_block_json = result
    stat_block = json.loads(stat_block_json)

    print(f"\n{name} (CR {cr})")
    print("=" * 40)
    print(f"AC: {stat_block['ac']}")
    print(f"HP: {stat_block['hp_avg']} ({stat_block['hp_dice']})")
    print(f"\nAbilities:")
    abilities = stat_block['abilities']
    for ability, score in abilities.items():
        modifier = (score - 10) // 2
        mod_str = f"+{modifier}" if modifier >= 0 else str(modifier)
        print(f"  {ability.upper()}: {score} ({mod_str})")

    attack = stat_block['attack']
    print(f"\nAttack: {attack['name']}")
    print(f"  To Hit: +{attack['bonus']}")
    print(f"  Damage: {attack['damage_dice']} {attack['damage_type']}")

    return 0

def list_monsters(max_cr=None):
    """List all monsters in the bestiary, optionally filtered by max CR."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    if max_cr is not None:
        cursor.execute("SELECT name, cr FROM bestiary WHERE cr <= ? ORDER BY cr", (max_cr,))
    else:
        cursor.execute("SELECT name, cr FROM bestiary ORDER BY cr")

    results = cursor.fetchall()
    conn.close()

    if not results:
        if max_cr is not None:
            print(f"No monsters found with CR ≤ {max_cr}")
        else:
            print("No monsters in bestiary")
        return 0

    print("Bestiary:")
    for name, cr in results:
        print(f"  • {name} (CR {cr})")

    return 0

def select_monster(max_cr):
    """Select an appropriate monster for the given max CR."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute(
        "SELECT name, cr, stat_block FROM bestiary WHERE cr <= ? ORDER BY cr DESC LIMIT 1",
        (max_cr,)
    )
    result = cursor.fetchone()
    conn.close()

    if not result:
        print(f"✗ Error: No monsters found with CR ≤ {max_cr}", file=sys.stderr)
        return 1

    name, cr, stat_block_json = result
    stat_block = json.loads(stat_block_json)

    # Output as JSON for easy parsing by combat system
    output = {
        "name": name,
        "cr": cr,
        **stat_block
    }
    print(json.dumps(output))
    return 0

def seed_bestiary():
    """Seed the bestiary with initial monsters."""
    init_db()

    monsters = [
        {
            "name": "Goblin",
            "cr": 0.25,
            "stat_block": {
                "ac": 15,
                "hp_dice": "2d6",
                "hp_avg": 7,
                "abilities": {
                    "str": 8,
                    "dex": 14,
                    "con": 10,
                    "int": 10,
                    "wis": 8,
                    "cha": 8
                },
                "attack": {
                    "name": "Scimitar",
                    "bonus": 4,
                    "damage_dice": "1d6+2",
                    "damage_type": "slashing"
                }
            }
        },
        {
            "name": "Skeleton",
            "cr": 0.25,
            "stat_block": {
                "ac": 13,
                "hp_dice": "2d8+4",
                "hp_avg": 13,
                "abilities": {
                    "str": 10,
                    "dex": 14,
                    "con": 15,
                    "int": 6,
                    "wis": 8,
                    "cha": 5
                },
                "attack": {
                    "name": "Shortsword",
                    "bonus": 4,
                    "damage_dice": "1d6+2",
                    "damage_type": "piercing"
                }
            }
        },
        {
            "name": "Orc",
            "cr": 0.5,
            "stat_block": {
                "ac": 13,
                "hp_dice": "2d8+6",
                "hp_avg": 15,
                "abilities": {
                    "str": 16,
                    "dex": 12,
                    "con": 16,
                    "int": 7,
                    "wis": 11,
                    "cha": 10
                },
                "attack": {
                    "name": "Greataxe",
                    "bonus": 5,
                    "damage_dice": "1d12+3",
                    "damage_type": "slashing"
                }
            }
        }
    ]

    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    seeded_count = 0
    for monster in monsters:
        try:
            cursor.execute(
                "INSERT INTO bestiary (name, cr, stat_block) VALUES (?, ?, ?)",
                (monster["name"], monster["cr"], json.dumps(monster["stat_block"]))
            )
            seeded_count += 1
        except sqlite3.IntegrityError:
            # Monster already exists, skip
            pass

    conn.commit()
    conn.close()

    if seeded_count > 0:
        print(f"✓ Seeded {seeded_count} monsters into bestiary")
    else:
        print("Bestiary already seeded")

    return 0


def export_monsters(monster_names):
    """Export monsters to JSON format for encounter.py."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    foes = []
    name_counts = {}

    for monster_name in monster_names:
        # Get monster from bestiary
        cursor.execute("SELECT name, cr, stat_block FROM bestiary WHERE name = ?", (monster_name,))
        result = cursor.fetchone()

        if not result:
            print(f"Error: Monster '{monster_name}' not found in bestiary", file=sys.stderr)
            conn.close()
            sys.exit(1)

        base_name, cr, stat_block_json = result
        stat_block = json.loads(stat_block_json)

        # Auto-number duplicates (Goblin → Goblin-1, Goblin-2, etc.)
        if base_name not in name_counts:
            name_counts[base_name] = 0

        name_counts[base_name] += 1

        # Use numbered name if multiple of same type
        if monster_names.count(base_name) > 1:
            numbered_name = f"{base_name}-{name_counts[base_name]}"
        else:
            numbered_name = base_name

        foe = {
            "name": numbered_name,
            "cr": cr,
            "hp_max": stat_block['hp_avg'],
            "hp_current": stat_block['hp_avg'],
            "ac": stat_block['ac'],
            "abilities": stat_block['abilities'],
            "attack": stat_block['attack']
        }

        foes.append(foe)

    conn.close()

    output = {"foes": foes}
    print(json.dumps(output, indent=2))
    return 0


def main():
    parser = argparse.ArgumentParser(description="Manage D&D monster bestiary")
    subparsers = parser.add_subparsers(dest="command", help="Command to execute")

    # Add command
    add_parser = subparsers.add_parser("add", help="Add a monster to the bestiary")
    add_parser.add_argument("name", help="Monster name")
    add_parser.add_argument("cr", type=float, help="Challenge Rating")
    add_parser.add_argument("stat_block", help="JSON stat block")

    # Show command
    show_parser = subparsers.add_parser("show", help="Show a monster's stat block")
    show_parser.add_argument("name", help="Monster name")

    # List command
    list_parser = subparsers.add_parser("list", help="List all monsters")
    list_parser.add_argument("--max-cr", type=float, help="Filter by max CR")

    # Select command
    select_parser = subparsers.add_parser("select", help="Select appropriate monster for max CR")
    select_parser.add_argument("max_cr", type=float, help="Maximum CR")

    # Seed command
    subparsers.add_parser("seed", help="Seed bestiary with initial monsters")

    # Export command
    export_parser = subparsers.add_parser("export", help="Export monsters to JSON for encounter.py")
    export_parser.add_argument("monsters", nargs='+', help="Monster names (duplicates auto-numbered)")

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        return 1

    if args.command == "add":
        return add_monster(args.name, args.cr, args.stat_block)
    elif args.command == "show":
        return show_monster(args.name)
    elif args.command == "list":
        return list_monsters(args.max_cr)
    elif args.command == "select":
        return select_monster(args.max_cr)
    elif args.command == "seed":
        return seed_bestiary()
    elif args.command == "export":
        return export_monsters(args.monsters)

    return 0

if __name__ == "__main__":
    sys.exit(main())
