#!/usr/bin/env python3
# ABOUTME: D&D character management script with SQLite persistence
# ABOUTME: Handles character CRUD operations, ability scores, HP, level tracking, and spell slots

import sqlite3
import argparse
import sys
import os
import json
from pathlib import Path
from datetime import datetime


# Database location
DB_PATH = Path.home() / ".claude" / "data" / "dnd-dm.db"


def init_db():
    """Initialize database and create characters table if it doesn't exist."""
    DB_PATH.parent.mkdir(parents=True, exist_ok=True)

    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS characters (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL UNIQUE,
            class TEXT NOT NULL,
            level INTEGER DEFAULT 1,
            xp INTEGER DEFAULT 0,
            party_id TEXT DEFAULT NULL,

            strength INTEGER NOT NULL,
            dexterity INTEGER NOT NULL,
            constitution INTEGER NOT NULL,
            intelligence INTEGER NOT NULL,
            wisdom INTEGER NOT NULL,
            charisma INTEGER NOT NULL,

            hp_current INTEGER NOT NULL,
            hp_max INTEGER NOT NULL,

            spell_slots TEXT DEFAULT NULL,

            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """)

    conn.commit()
    conn.close()


def calculate_modifier(score):
    """Calculate ability modifier from ability score."""
    return (score - 10) // 2


def format_modifier(modifier):
    """Format modifier with + or - sign."""
    return f"+{modifier}" if modifier >= 0 else str(modifier)


def calculate_hp(constitution_score, character_class, level):
    """Calculate max HP based on class, level, and CON."""
    # Hit dice by class (using Fighter as default)
    hit_dice = {
        'fighter': 10,
        'paladin': 10,
        'ranger': 10,
        'barbarian': 12,
        'wizard': 6,
        'sorcerer': 6,
        'rogue': 8,
        'bard': 8,
        'cleric': 8,
        'druid': 8,
        'monk': 8,
        'warlock': 8
    }

    hd = hit_dice.get(character_class.lower(), 8)
    con_mod = calculate_modifier(constitution_score)

    # Level 1: max hit die + CON mod
    # Additional levels: average of hit die + CON mod per level
    if level == 1:
        return hd + con_mod
    else:
        avg_roll = (hd // 2) + 1
        return hd + con_mod + ((avg_roll + con_mod) * (level - 1))


def calculate_spell_slots(character_class, level):
    """Calculate spell slots for a character based on class and level."""
    # Spell slot progression for full casters (level 1 = 2 slots)
    full_casters = ['wizard', 'sorcerer', 'cleric', 'druid', 'warlock']

    char_class_lower = character_class.lower()

    if char_class_lower in full_casters:
        if level == 1:
            return {"1": {"max": 2, "current": 2}}
        # For future expansion: higher levels

    # Non-casters or classes that don't get spells at level 1
    return None


def initialize_spell_slots(character_class, level):
    """Initialize spell slots for a new character."""
    slots = calculate_spell_slots(character_class, level)
    return json.dumps(slots) if slots else None


def restore_spell_slots(spell_slots_json):
    """Restore all spell slots to max (long rest)."""
    if not spell_slots_json:
        return None

    slots = json.loads(spell_slots_json)
    for level_key in slots:
        slots[level_key]["current"] = slots[level_key]["max"]

    return json.dumps(slots)


def use_spell_slot(spell_slots_json, slot_level):
    """Use a spell slot of the given level. Returns updated JSON or None if no slots available."""
    if not spell_slots_json:
        return None, False

    slots = json.loads(spell_slots_json)
    level_key = str(slot_level)

    if level_key not in slots:
        return spell_slots_json, False

    if slots[level_key]["current"] > 0:
        slots[level_key]["current"] -= 1
        return json.dumps(slots), True

    return spell_slots_json, False


def create_character(args):
    """Create a new character."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Calculate HP
    hp_max = calculate_hp(args.con, args.char_class, args.level)
    hp_current = hp_max if args.hp is None else args.hp

    # Initialize spell slots
    spell_slots = initialize_spell_slots(args.char_class, args.level)

    # Get party_id if provided
    party_id = getattr(args, 'party', None)

    try:
        cursor.execute("""
            INSERT INTO characters
            (name, class, level, xp, party_id, strength, dexterity, constitution,
             intelligence, wisdom, charisma, hp_current, hp_max, spell_slots)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        """, (args.name, args.char_class, args.level, 0, party_id,
              args.str, args.dex, args.con, args.int, args.wis, args.cha,
              hp_current, hp_max, spell_slots))

        conn.commit()
        print(f"✓ Created {args.name} ({args.char_class}, Level {args.level})")

        # Show the character
        show_character(args.name, cursor)

    except sqlite3.IntegrityError:
        print(f"Error: Character '{args.name}' already exists")
        sys.exit(1)
    finally:
        conn.close()


def show_character(name, cursor=None):
    """Display a character's stats."""
    should_close = cursor is None

    if cursor is None:
        init_db()
        conn = sqlite3.connect(DB_PATH)
        cursor = conn.cursor()

    cursor.execute("""
        SELECT name, class, level, xp, strength, dexterity, constitution,
               intelligence, wisdom, charisma, hp_current, hp_max, spell_slots
        FROM characters WHERE name = ?
    """, (name,))
    char = cursor.fetchone()

    if should_close:
        conn.close()

    if not char:
        print(f"Error: Character '{name}' not found")
        sys.exit(1)

    # Unpack character data
    (name, char_class, level, xp, str_score, dex_score, con_score,
     int_score, wis_score, cha_score, hp_current, hp_max, spell_slots_json) = char

    # Calculate modifiers
    str_mod = calculate_modifier(str_score)
    dex_mod = calculate_modifier(dex_score)
    con_mod = calculate_modifier(con_score)
    int_mod = calculate_modifier(int_score)
    wis_mod = calculate_modifier(wis_score)
    cha_mod = calculate_modifier(cha_score)

    # Proficiency bonus by level
    prof_bonus = 2 + ((level - 1) // 4)

    # XP thresholds for levels 1-20
    xp_thresholds = [0, 300, 900, 2700, 6500, 14000, 23000, 34000, 48000, 64000,
                     85000, 100000, 120000, 140000, 165000, 195000, 225000, 265000, 305000, 355000]

    xp_needed = xp_thresholds[level] if level < 20 else "MAX"

    print(f"\n{name} ({char_class}, Level {level})")
    print("━" * 30)
    if level < 20:
        print(f"XP: {xp}/{xp_needed} (need {xp_needed - xp} more)")
    else:
        print(f"XP: {xp} (MAX LEVEL)")
    print()
    print(f"STR: {str_score:2d} ({format_modifier(str_mod)})")
    print(f"DEX: {dex_score:2d} ({format_modifier(dex_mod)})")
    print(f"CON: {con_score:2d} ({format_modifier(con_mod)})")
    print(f"INT: {int_score:2d} ({format_modifier(int_mod)})")
    print(f"WIS: {wis_score:2d} ({format_modifier(wis_mod)})")
    print(f"CHA: {cha_score:2d} ({format_modifier(cha_mod)})")
    print()
    print(f"HP: {hp_current}/{hp_max}")
    print(f"Proficiency: +{prof_bonus}")

    # Display spell slots if character is a caster
    if spell_slots_json:
        spell_slots = json.loads(spell_slots_json)
        print()
        print("Spell Slots:")
        for level_key, slots in sorted(spell_slots.items()):
            level_name = "Cantrips" if level_key == "0" else f"Level {level_key}"
            print(f"  {level_name}: {slots['current']}/{slots['max']}")

    print()


def list_characters(args):
    """List all characters."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    if args.party:
        cursor.execute(
            "SELECT name, class, level, party_id FROM characters WHERE party_id = ? ORDER BY name",
            (args.party,)
        )
    elif args.char_class:
        cursor.execute(
            "SELECT name, class, level, party_id FROM characters WHERE class = ? ORDER BY name",
            (args.char_class,)
        )
    else:
        cursor.execute("SELECT name, class, level, party_id FROM characters ORDER BY name")

    characters = cursor.fetchall()
    conn.close()

    if not characters:
        print("No characters found")
        return

    print("\nCharacters:")
    for name, char_class, level, party_id in characters:
        party_str = f", Party: {party_id}" if party_id else ""
        print(f"  • {name} ({char_class}, Level {level}{party_str})")
    print()


def update_character(args):
    """Update a character's stats."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Check if character exists
    cursor.execute("SELECT * FROM characters WHERE name = ?", (args.name,))
    char = cursor.fetchone()

    if not char:
        print(f"Error: Character '{args.name}' not found")
        conn.close()
        sys.exit(1)

    # Build update query dynamically
    updates = []
    values = []

    if args.str is not None:
        updates.append("strength = ?")
        values.append(args.str)
    if args.dex is not None:
        updates.append("dexterity = ?")
        values.append(args.dex)
    if args.con is not None:
        updates.append("constitution = ?")
        values.append(args.con)
    if args.int is not None:
        updates.append("intelligence = ?")
        values.append(args.int)
    if args.wis is not None:
        updates.append("wisdom = ?")
        values.append(args.wis)
    if args.cha is not None:
        updates.append("charisma = ?")
        values.append(args.cha)
    if args.level is not None:
        updates.append("level = ?")
        values.append(args.level)
    if args.hp is not None:
        updates.append("hp_current = ?")
        values.append(args.hp)

    if not updates:
        print("No updates specified")
        conn.close()
        return

    # Add updated timestamp
    updates.append("updated_at = ?")
    values.append(datetime.now().isoformat())

    values.append(args.name)

    query = f"UPDATE characters SET {', '.join(updates)} WHERE name = ?"
    cursor.execute(query, values)
    conn.commit()

    print(f"✓ Updated {args.name}")
    show_character(args.name, cursor)

    conn.close()


def delete_character(args):
    """Delete a character."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("DELETE FROM characters WHERE name = ?", (args.name,))

    if cursor.rowcount == 0:
        print(f"Error: Character '{args.name}' not found")
    else:
        print(f"✓ Deleted {args.name}")

    conn.commit()
    conn.close()


def show_info(args):
    """Show database information."""
    print(f"Database location: {DB_PATH}")
    print(f"Database exists: {DB_PATH.exists()}")

    if DB_PATH.exists():
        size = DB_PATH.stat().st_size
        print(f"Database size: {size} bytes")

        conn = sqlite3.connect(DB_PATH)
        cursor = conn.cursor()
        cursor.execute("SELECT COUNT(*) FROM characters")
        count = cursor.fetchone()[0]
        print(f"Total characters: {count}")
        conn.close()


def long_rest(args):
    """Restore character's HP and spell slots (long rest)."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Get current character data
    cursor.execute("SELECT hp_max, spell_slots FROM characters WHERE name = ?", (args.name,))
    result = cursor.fetchone()

    if not result:
        print(f"Error: Character '{args.name}' not found")
        conn.close()
        sys.exit(1)

    hp_max, spell_slots_json = result

    # Restore spell slots
    restored_slots = restore_spell_slots(spell_slots_json)

    # Update character
    cursor.execute(
        "UPDATE characters SET hp_current = ?, spell_slots = ?, updated_at = ? WHERE name = ?",
        (hp_max, restored_slots, datetime.now(), args.name)
    )

    conn.commit()
    conn.close()

    print(f"✓ {args.name} takes a long rest")
    print(f"  HP restored to {hp_max}")
    if restored_slots:
        print(f"  Spell slots restored")


def main():
    parser = argparse.ArgumentParser(description='D&D Character Management')
    subparsers = parser.add_subparsers(dest='command', help='Commands')

    # Create character
    create_parser = subparsers.add_parser('create', help='Create a new character')
    create_parser.add_argument('name', help='Character name')
    create_parser.add_argument('char_class', help='Character class')
    create_parser.add_argument('--str', type=int, required=True, help='Strength score')
    create_parser.add_argument('--dex', type=int, required=True, help='Dexterity score')
    create_parser.add_argument('--con', type=int, required=True, help='Constitution score')
    create_parser.add_argument('--int', type=int, required=True, help='Intelligence score')
    create_parser.add_argument('--wis', type=int, required=True, help='Wisdom score')
    create_parser.add_argument('--cha', type=int, required=True, help='Charisma score')
    create_parser.add_argument('--level', type=int, default=1, help='Character level (default: 1)')
    create_parser.add_argument('--hp', type=int, help='Current HP (default: calculated max HP)')
    create_parser.add_argument('--party', help='Party ID to assign character to')

    # Show character
    show_parser = subparsers.add_parser('show', help='Show character stats')
    show_parser.add_argument('name', help='Character name')

    # List characters
    list_parser = subparsers.add_parser('list', help='List all characters')
    list_parser.add_argument('--class', dest='char_class', help='Filter by class')
    list_parser.add_argument('--party', help='Filter by party ID')

    # Update character
    update_parser = subparsers.add_parser('update', help='Update character stats')
    update_parser.add_argument('name', help='Character name')
    update_parser.add_argument('--str', type=int, help='Update Strength')
    update_parser.add_argument('--dex', type=int, help='Update Dexterity')
    update_parser.add_argument('--con', type=int, help='Update Constitution')
    update_parser.add_argument('--int', type=int, help='Update Intelligence')
    update_parser.add_argument('--wis', type=int, help='Update Wisdom')
    update_parser.add_argument('--cha', type=int, help='Update Charisma')
    update_parser.add_argument('--level', type=int, help='Update level')
    update_parser.add_argument('--hp', type=int, help='Update current HP')

    # Delete character
    delete_parser = subparsers.add_parser('delete', help='Delete a character')
    delete_parser.add_argument('name', help='Character name')

    # Info
    info_parser = subparsers.add_parser('info', help='Show database information')

    # Long rest
    rest_parser = subparsers.add_parser('long-rest', help='Take a long rest (restore HP and spell slots)')
    rest_parser.add_argument('name', help='Character name')

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        sys.exit(1)

    if args.command == 'create':
        create_character(args)
    elif args.command == 'show':
        show_character(args.name)
    elif args.command == 'list':
        list_characters(args)
    elif args.command == 'update':
        update_character(args)
    elif args.command == 'delete':
        delete_character(args)
    elif args.command == 'info':
        show_info(args)
    elif args.command == 'long-rest':
        long_rest(args)


if __name__ == '__main__':
    main()
