#!/usr/bin/env python3
# ABOUTME: Initiative and turn order management for D&D combat
# ABOUTME: Implements priority queue for combat turn ordering based on initiative rolls

import sqlite3
import json
import argparse
import sys
import subprocess
from pathlib import Path

# Database location
DB_PATH = Path.home() / ".claude" / "data" / "dnd-dm.db"
SCRIPTS_DIR = Path(__file__).parent


def init_db():
    """Initialize database and create initiative table if it doesn't exist."""
    DB_PATH.parent.mkdir(parents=True, exist_ok=True)

    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS initiative (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            combat_id TEXT NOT NULL,
            combatant_name TEXT NOT NULL,
            combatant_type TEXT NOT NULL,
            initiative_roll INTEGER NOT NULL,
            is_active INTEGER DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """)

    # Create index for faster combat_id lookups
    cursor.execute("""
        CREATE INDEX IF NOT EXISTS idx_combat_id ON initiative(combat_id)
    """)

    conn.commit()
    conn.close()


def roll_dice(dice_expr):
    """Roll dice using roll_dice.py script."""
    result = subprocess.run(
        [sys.executable, str(SCRIPTS_DIR / "roll_dice.py"), dice_expr],
        capture_output=True,
        text=True,
        check=True
    )
    # Parse output like "1d20: [15] = 15"
    output = result.stdout.strip()
    total = int(output.split("=")[-1].strip())
    return total


def get_dex_modifier(combatant_name, combatant_type):
    """Get DEX modifier for a combatant."""
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    if combatant_type == "character":
        cursor.execute("SELECT dexterity FROM characters WHERE name = ?", (combatant_name,))
    else:  # monster
        cursor.execute("SELECT stat_block FROM bestiary WHERE name = ?", (combatant_name,))
        result = cursor.fetchone()
        if result:
            stat_block = json.loads(result[0])
            dex_score = stat_block['abilities']['dex']
            conn.close()
            return (dex_score - 10) // 2
        conn.close()
        return 0

    result = cursor.fetchone()
    conn.close()

    if result:
        dex_score = result[0]
        return (dex_score - 10) // 2

    return 0


def roll_initiative(combat_id, combatants):
    """
    Roll initiative for all combatants and insert into initiative table.
    combatants: list of (name, type) tuples where type is 'character' or 'monster'
    Returns: list of (name, type, initiative) sorted by initiative DESC
    """
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Clear any existing initiative for this combat
    cursor.execute("DELETE FROM initiative WHERE combat_id = ?", (combat_id,))

    results = []

    for name, combatant_type in combatants:
        # Roll 1d20
        d20_roll = roll_dice("1d20")

        # Get DEX modifier
        dex_mod = get_dex_modifier(name, combatant_type)

        # Initiative = 1d20 + DEX
        initiative = d20_roll + dex_mod

        # Insert into database
        cursor.execute("""
            INSERT INTO initiative (combat_id, combatant_name, combatant_type, initiative_roll)
            VALUES (?, ?, ?, ?)
        """, (combat_id, name, combatant_type, initiative))

        results.append((name, combatant_type, initiative, d20_roll, dex_mod))

    conn.commit()
    conn.close()

    # Sort by initiative DESC (highest first), then by DEX modifier DESC for ties
    results.sort(key=lambda x: (x[2], x[4]), reverse=True)

    return results


def get_turn_order(combat_id):
    """
    Get current turn order for a combat.
    Returns: list of (name, type, initiative, is_active) sorted by initiative DESC
    """
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("""
        SELECT combatant_name, combatant_type, initiative_roll, is_active
        FROM initiative
        WHERE combat_id = ?
        ORDER BY initiative_roll DESC, combatant_name
    """, (combat_id,))

    results = cursor.fetchall()
    conn.close()

    return results


def mark_defeated(combat_id, combatant_name):
    """Mark a combatant as defeated (is_active = 0)."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("""
        UPDATE initiative
        SET is_active = 0
        WHERE combat_id = ? AND combatant_name = ?
    """, (combat_id, combatant_name))

    rows_updated = cursor.rowcount
    conn.commit()
    conn.close()

    if rows_updated == 0:
        print(f"Warning: {combatant_name} not found in initiative for combat '{combat_id}'")
        return False

    return True


def show_initiative(args):
    """Display initiative order for a combat."""
    turn_order = get_turn_order(args.combat_id)

    if not turn_order:
        print(f"No initiative data for combat '{args.combat_id}'")
        return

    print(f"\nInitiative Order - Combat '{args.combat_id}'")
    print("━" * 50)

    for i, (name, combatant_type, initiative, is_active) in enumerate(turn_order, 1):
        status = "✓" if is_active else "✗ DEFEATED"
        type_icon = "👤" if combatant_type == "character" else "👹"
        print(f"  {i:2}. [{initiative:2}] {type_icon} {name:20} {status}")

    print()


def roll_cmd(args):
    """Roll initiative for combatants."""
    # Parse combatants from args
    # Format: "Name:type Name:type ..."
    combatants = []
    for spec in args.combatants:
        if ':' not in spec:
            print(f"Error: Invalid combatant spec '{spec}' (expected 'Name:type')")
            sys.exit(1)

        name, combatant_type = spec.split(':', 1)

        if combatant_type not in ['character', 'monster']:
            print(f"Error: Invalid combatant type '{combatant_type}' (expected 'character' or 'monster')")
            sys.exit(1)

        combatants.append((name, combatant_type))

    # Roll initiative
    results = roll_initiative(args.combat_id, combatants)

    print(f"\nInitiative Rolled - Combat '{args.combat_id}'")
    print("━" * 60)

    for name, combatant_type, initiative, d20_roll, dex_mod in results:
        type_str = "Character" if combatant_type == "character" else "Monster"
        dex_str = f"+{dex_mod}" if dex_mod >= 0 else str(dex_mod)
        print(f"  [{initiative:2}] {name:20} ({type_str}) - d20: {d20_roll}, DEX: {dex_str}")

    print()


def defeat_cmd(args):
    """Mark a combatant as defeated."""
    success = mark_defeated(args.combat_id, args.combatant_name)

    if success:
        print(f"✓ Marked {args.combatant_name} as defeated")


def clear_cmd(args):
    """Clear all initiative data for a combat."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("DELETE FROM initiative WHERE combat_id = ?", (args.combat_id,))
    rows_deleted = cursor.rowcount

    conn.commit()
    conn.close()

    if rows_deleted > 0:
        print(f"✓ Cleared initiative data for combat '{args.combat_id}' ({rows_deleted} combatants)")
    else:
        print(f"No initiative data found for combat '{args.combat_id}'")


def main():
    parser = argparse.ArgumentParser(description='D&D Initiative Management')
    subparsers = parser.add_subparsers(dest='command', help='Commands')

    # Roll initiative
    roll_parser = subparsers.add_parser('roll', help='Roll initiative for combatants')
    roll_parser.add_argument('combat_id', help='Combat ID (e.g., "combat-1")')
    roll_parser.add_argument('combatants', nargs='+', help='Combatants as "Name:type" (type = character|monster)')

    # Show initiative
    show_parser = subparsers.add_parser('show', help='Show initiative order')
    show_parser.add_argument('combat_id', help='Combat ID')

    # Mark defeated
    defeat_parser = subparsers.add_parser('defeat', help='Mark combatant as defeated')
    defeat_parser.add_argument('combat_id', help='Combat ID')
    defeat_parser.add_argument('combatant_name', help='Combatant name')

    # Clear initiative
    clear_parser = subparsers.add_parser('clear', help='Clear initiative for combat')
    clear_parser.add_argument('combat_id', help='Combat ID')

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        sys.exit(1)

    if args.command == 'roll':
        roll_cmd(args)
    elif args.command == 'show':
        show_initiative(args)
    elif args.command == 'defeat':
        defeat_cmd(args)
    elif args.command == 'clear':
        clear_cmd(args)


if __name__ == '__main__':
    main()
