#!/usr/bin/env python3
# ABOUTME: Party management script for D&D groups
# ABOUTME: Handles creating parties, adding/removing members, and displaying party composition

import sqlite3
import argparse
import sys
import json
from pathlib import Path

# Database location
DB_PATH = Path.home() / ".claude" / "data" / "dnd-dm.db"


def init_db():
    """Initialize database and create parties table if it doesn't exist."""
    DB_PATH.parent.mkdir(parents=True, exist_ok=True)

    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Parties table for tracking party metadata
    cursor.execute("""
        CREATE TABLE IF NOT EXISTS parties (
            id TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            description TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """)

    conn.commit()
    conn.close()


def create_party(args):
    """Create a new party."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    try:
        cursor.execute("""
            INSERT INTO parties (id, name, description)
            VALUES (?, ?, ?)
        """, (args.id, args.name, args.description or ""))

        conn.commit()
        print(f"✓ Created party '{args.name}' (ID: {args.id})")

    except sqlite3.IntegrityError:
        print(f"Error: Party with ID '{args.id}' already exists")
        sys.exit(1)
    finally:
        conn.close()


def add_member(args):
    """Add a character to a party."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Check if party exists
    cursor.execute("SELECT name FROM parties WHERE id = ?", (args.party_id,))
    party = cursor.fetchone()

    if not party:
        print(f"Error: Party '{args.party_id}' not found")
        conn.close()
        sys.exit(1)

    # Check if character exists
    cursor.execute("SELECT name, party_id FROM characters WHERE name = ?", (args.character_name,))
    char = cursor.fetchone()

    if not char:
        print(f"Error: Character '{args.character_name}' not found")
        conn.close()
        sys.exit(1)

    char_name, current_party = char

    if current_party:
        print(f"Warning: {char_name} is already in party '{current_party}'")
        if not args.force:
            print("Use --force to reassign")
            conn.close()
            sys.exit(1)

    # Add character to party
    cursor.execute(
        "UPDATE characters SET party_id = ? WHERE name = ?",
        (args.party_id, args.character_name)
    )

    conn.commit()
    conn.close()

    print(f"✓ Added {args.character_name} to party '{party[0]}'")


def remove_member(args):
    """Remove a character from their party."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Check if character exists
    cursor.execute("SELECT name, party_id FROM characters WHERE name = ?", (args.character_name,))
    char = cursor.fetchone()

    if not char:
        print(f"Error: Character '{args.character_name}' not found")
        conn.close()
        sys.exit(1)

    char_name, current_party = char

    if not current_party:
        print(f"Warning: {char_name} is not in any party")
        conn.close()
        return

    # Remove from party
    cursor.execute(
        "UPDATE characters SET party_id = NULL WHERE name = ?",
        (args.character_name,)
    )

    conn.commit()
    conn.close()

    print(f"✓ Removed {char_name} from party '{current_party}'")


def show_party(args):
    """Display party composition and stats."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Get party info
    cursor.execute("SELECT id, name, description FROM parties WHERE id = ?", (args.party_id,))
    party = cursor.fetchone()

    if not party:
        print(f"Error: Party '{args.party_id}' not found")
        conn.close()
        sys.exit(1)

    party_id, party_name, description = party

    # Get party members
    cursor.execute("""
        SELECT name, class, level, hp_current, hp_max
        FROM characters
        WHERE party_id = ?
        ORDER BY name
    """, (args.party_id,))

    members = cursor.fetchall()
    conn.close()

    print(f"\n{party_name} (ID: {party_id})")
    if description:
        print(f"{description}")
    print("━" * 50)

    if not members:
        print("  (No members)")
        return

    print(f"\nMembers ({len(members)}):")
    for name, char_class, level, hp_current, hp_max in members:
        hp_pct = (hp_current / hp_max * 100) if hp_max > 0 else 0
        hp_status = "■" * int(hp_pct / 10) + "□" * (10 - int(hp_pct / 10))
        print(f"  • {name:12} {char_class:10} Level {level:2}  HP: {hp_current:3}/{hp_max:3} [{hp_status}]")

    print()


def list_parties(args):
    """List all parties."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute("SELECT id, name, description FROM parties ORDER BY name")
    parties = cursor.fetchall()

    if not parties:
        print("No parties found")
        conn.close()
        return

    print("\nParties:")
    for party_id, name, description in parties:
        # Count members
        cursor.execute("SELECT COUNT(*) FROM characters WHERE party_id = ?", (party_id,))
        member_count = cursor.fetchone()[0]

        desc_str = f" - {description}" if description else ""
        print(f"  • {name} (ID: {party_id}) - {member_count} member(s){desc_str}")

    print()
    conn.close()


def export_party(args):
    """Export party to JSON format for encounter.py."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Get party members
    cursor.execute("""
        SELECT name, class, level, strength, dexterity, constitution,
               intelligence, wisdom, charisma, hp_current, hp_max
        FROM characters
        WHERE party_id = ?
        ORDER BY name
    """, (args.party_id,))

    members = cursor.fetchall()
    conn.close()

    if not members:
        print(f"Error: No members found in party '{args.party_id}'", file=sys.stderr)
        sys.exit(1)

    # Build heroes JSON
    heroes = []
    for (name, char_class, level, str_score, dex_score, con_score,
         int_score, wis_score, cha_score, hp_current, hp_max) in members:

        # Calculate AC (simplified - assume 10 + DEX for now, equipment.py would provide real AC)
        dex_mod = (dex_score - 10) // 2
        ac = 10 + dex_mod  # Default unarmored

        hero = {
            "name": name,
            "class": char_class,
            "level": level,
            "hp_max": hp_max,
            "hp_current": hp_current,
            "ac": ac,
            "abilities": {
                "str": str_score,
                "dex": dex_score,
                "con": con_score,
                "int": int_score,
                "wis": wis_score,
                "cha": cha_score
            }
        }
        heroes.append(hero)

    output = {"heroes": heroes}
    print(json.dumps(output, indent=2))


def delete_party(args):
    """Delete a party (members remain, party_id set to NULL)."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Check if party exists
    cursor.execute("SELECT name FROM parties WHERE id = ?", (args.party_id,))
    party = cursor.fetchone()

    if not party:
        print(f"Error: Party '{args.party_id}' not found")
        conn.close()
        sys.exit(1)

    # Remove party_id from all members
    cursor.execute("UPDATE characters SET party_id = NULL WHERE party_id = ?", (args.party_id,))

    # Delete party
    cursor.execute("DELETE FROM parties WHERE id = ?", (args.party_id,))

    conn.commit()
    conn.close()

    print(f"✓ Deleted party '{party[0]}' (members remain, unassigned)")


def main():
    parser = argparse.ArgumentParser(description='D&D Party Management')
    subparsers = parser.add_subparsers(dest='command', help='Commands')

    # Create party
    create_parser = subparsers.add_parser('create', help='Create a new party')
    create_parser.add_argument('id', help='Party ID (unique identifier)')
    create_parser.add_argument('name', help='Party name')
    create_parser.add_argument('--description', help='Party description')

    # Add member
    add_parser = subparsers.add_parser('add', help='Add character to party')
    add_parser.add_argument('party_id', help='Party ID')
    add_parser.add_argument('character_name', help='Character name')
    add_parser.add_argument('--force', action='store_true', help='Force reassignment if already in party')

    # Remove member
    remove_parser = subparsers.add_parser('remove', help='Remove character from party')
    remove_parser.add_argument('character_name', help='Character name')

    # Show party
    show_parser = subparsers.add_parser('show', help='Show party composition')
    show_parser.add_argument('party_id', help='Party ID')

    # List parties
    list_parser = subparsers.add_parser('list', help='List all parties')

    # Delete party
    delete_parser = subparsers.add_parser('delete', help='Delete a party')
    delete_parser.add_argument('party_id', help='Party ID')

    # Export party
    export_parser = subparsers.add_parser('export', help='Export party to JSON for encounter.py')
    export_parser.add_argument('party_id', help='Party ID')

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        sys.exit(1)

    if args.command == 'create':
        create_party(args)
    elif args.command == 'add':
        add_member(args)
    elif args.command == 'remove':
        remove_member(args)
    elif args.command == 'show':
        show_party(args)
    elif args.command == 'list':
        list_parties(args)
    elif args.command == 'delete':
        delete_party(args)
    elif args.command == 'export':
        export_party(args)


if __name__ == '__main__':
    main()
