/**
 * Visual Snapshot Extension
 *
 * Adds image comparison capabilities.
 */
import { StdioServerTransport } from "@modelcontextprotocol/sdk/server/stdio.js";
import { Server } from "@modelcontextprotocol/sdk/server/index.js";
import {
  CallToolRequestSchema,
  ListToolsRequestSchema,
} from "@modelcontextprotocol/sdk/types.js";
import fs from "fs";
import { PNG } from "pngjs";
import pixelmatch from "pixelmatch";
import path from "path";

const server = new Server(
  {
    name: "visual-snapshot-local",
    version: "1.0.0",
  },
  {
    capabilities: {
      tools: {},
    },
  },
);

function compareImages(img1Path, img2Path) {
  const img1 = PNG.sync.read(fs.readFileSync(img1Path));
  const img2 = PNG.sync.read(fs.readFileSync(img2Path));
  const { width, height } = img1;
  const diff = new PNG({ width, height });

  const numDiffPixels = pixelmatch(
    img1.data,
    img2.data,
    diff.data,
    width,
    height,
    { threshold: 0.1 },
  );

  const diffPath = path.join(path.dirname(img2Path), `diff-${Date.now()}.png`);
  fs.writeFileSync(diffPath, PNG.sync.write(diff));

  return {
    match: numDiffPixels === 0,
    diffPixels: numDiffPixels,
    diffPercentage: (numDiffPixels / (width * height)) * 100,
    diffImage: diffPath,
  };
}

server.setRequestHandler(ListToolsRequestSchema, async () => {
  return {
    tools: [
      {
        name: "compare_images",
        description: "Compare two images and return the difference.",
        inputSchema: {
          type: "object",
          properties: {
            baseline_path: { type: "string" },
            current_path: { type: "string" },
          },
          required: ["baseline_path", "current_path"],
        },
      },
    ],
  };
});

server.setRequestHandler(CallToolRequestSchema, async (request) => {
  if (request.params.name === "compare_images") {
    const { baseline_path, current_path } = request.params.arguments;
    try {
      const result = compareImages(baseline_path, current_path);
      return {
        content: [{ type: "text", text: JSON.stringify(result, null, 2) }],
      };
    } catch (error) {
      return {
        isError: true,
        content: [{ type: "text", text: `Error: ${error.message}` }],
      };
    }
  }
  throw new Error("Tool not found");
});

async function run() {
  const transport = new StdioServerTransport();
  await server.connect(transport);
}

run().catch(console.error);
