# Watcher Management Examples

## Example 1: Gmail Watcher

```python
# gmail_watcher.py
import time
import logging
from pathlib import Path
from datetime import datetime
from google.oauth2.credentials import Credentials
from googleapiclient.discovery import build
import base64

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger('GmailWatcher')

class GmailWatcher:
    def __init__(self, vault_path: str, credentials_path: str, token_path: str):
        self.vault_path = Path(vault_path)
        self.needs_action = self.vault_path / 'Needs_Action'
        self.needs_action.mkdir(exist_ok=True)

        # Load credentials
        self.creds = Credentials.from_authorized_user_file(token_path)
        self.service = build('gmail', 'v1', credentials=self.creds)

        # Track processed messages
        self.processed_ids = set()
        self.check_interval = 120  # 2 minutes

    def check_for_updates(self):
        """Check for new unread important emails."""
        try:
            results = self.service.users().messages().list(
                userId='me',
                q='is:unread is:important',
                maxResults=10
            ).execute()

            messages = results.get('messages', [])
            new_messages = [m for m in messages if m['id'] not in self.processed_ids]

            logger.info(f"Found {len(new_messages)} new messages")
            return new_messages

        except Exception as e:
            logger.error(f"Error checking emails: {e}")
            return []

    def get_email_content(self, message_id: str) -> dict:
        """Fetch full email content."""
        msg = self.service.users().messages().get(
            userId='me',
            id=message_id,
            format='full'
        ).execute()

        headers = {h['name']: h['value'] for h in msg['payload']['headers']}

        # Get body
        body = ''
        if 'parts' in msg['payload']:
            for part in msg['payload']['parts']:
                if part['mimeType'] == 'text/plain':
                    body = base64.urlsafe_b64decode(
                        part['body'].get('data', '')
                    ).decode('utf-8', errors='ignore')
                    break
        else:
            body = msg.get('snippet', '')

        return {
            'id': message_id,
            'from': headers.get('From', 'Unknown'),
            'to': headers.get('To', ''),
            'subject': headers.get('Subject', 'No Subject'),
            'date': headers.get('Date', ''),
            'body': body,
            'snippet': msg.get('snippet', ''),
            'labels': msg.get('labelIds', [])
        }

    def create_action_file(self, email: dict):
        """Create an action file for the email."""
        # Determine priority
        priority = 'high' if 'IMPORTANT' in email.get('labels', []) else 'medium'

        content = f"""---
type: email
from: {email['from']}
to: {email['to']}
subject: {email['subject']}
received: {datetime.now().isoformat()}
message_id: {email['id']}
priority: {priority}
status: pending
---

## Email Content
{email['body'][:2000]}

## Snippet
{email['snippet']}

## Suggested Actions
- [ ] Reply to sender
- [ ] Forward to relevant party
- [ ] Archive after processing
"""

        filename = f"EMAIL_{email['id']}.md"
        filepath = self.needs_action / filename
        filepath.write_text(content)

        logger.info(f"Created: {filename}")
        self.processed_ids.add(email['id'])

    def run(self):
        """Main run loop."""
        logger.info(f"Starting Gmail Watcher")
        logger.info(f"Vault: {self.vault_path}")
        logger.info(f"Check interval: {self.check_interval}s")

        while True:
            try:
                messages = self.check_for_updates()
                for msg in messages:
                    email = self.get_email_content(msg['id'])
                    self.create_action_file(email)

            except Exception as e:
                logger.error(f"Error in main loop: {e}")

            time.sleep(self.check_interval)


if __name__ == '__main__':
    import os

    vault = os.environ.get('VAULT_PATH', '/path/to/vault')
    creds = os.environ.get('GMAIL_CREDENTIALS')
    token = os.environ.get('GMAIL_TOKEN')

    watcher = GmailWatcher(vault, creds, token)
    watcher.run()
```

## Example 2: Filesystem Watcher

```python
# filesystem_watcher.py
import time
import logging
import shutil
from pathlib import Path
from datetime import datetime
from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger('FileWatcher')

class DropFolderHandler(FileSystemEventHandler):
    def __init__(self, vault_path: str, watch_extensions: list = None):
        self.vault_path = Path(vault_path)
        self.needs_action = self.vault_path / 'Needs_Action'
        self.needs_action.mkdir(exist_ok=True)

        self.watch_extensions = watch_extensions or ['.pdf', '.csv', '.xlsx', '.doc', '.docx']

    def on_created(self, event):
        """Handle new file creation."""
        if event.is_directory:
            return

        source = Path(event.src_path)

        # Check extension
        if source.suffix.lower() not in self.watch_extensions:
            logger.debug(f"Ignoring {source.name} (unsupported extension)")
            return

        logger.info(f"New file detected: {source.name}")

        # Copy file to Needs_Action
        dest_file = self.needs_action / f"FILE_{source.name}"
        shutil.copy2(source, dest_file)

        # Create metadata file
        self.create_metadata(source, dest_file)

    def create_metadata(self, source: Path, dest: Path):
        """Create a metadata markdown file."""
        content = f"""---
type: file_drop
original_name: {source.name}
original_path: {source}
size_bytes: {source.stat().st_size}
created: {datetime.now().isoformat()}
extension: {source.suffix}
status: pending
---

## File Information
- **Name**: {source.name}
- **Size**: {source.stat().st_size:,} bytes
- **Type**: {source.suffix}

## Processing Notes
New file dropped for processing.

## Suggested Actions
- [ ] Review file contents
- [ ] Categorize appropriately
- [ ] Process or archive
"""

        meta_path = dest.with_suffix('.md')
        meta_path.write_text(content)
        logger.info(f"Created metadata: {meta_path.name}")


def run_filesystem_watcher(vault_path: str, drop_folder: str):
    """Run the filesystem watcher."""
    logger.info(f"Starting Filesystem Watcher")
    logger.info(f"Watching: {drop_folder}")
    logger.info(f"Output to: {vault_path}/Needs_Action/")

    handler = DropFolderHandler(vault_path)
    observer = Observer()
    observer.schedule(handler, drop_folder, recursive=False)
    observer.start()

    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        observer.stop()

    observer.join()


if __name__ == '__main__':
    import os

    vault = os.environ.get('VAULT_PATH', '/path/to/vault')
    drop = os.environ.get('DROP_FOLDER', '/path/to/drop')

    run_filesystem_watcher(vault, drop)
```

## Example 3: WhatsApp Watcher (Playwright)

```python
# whatsapp_watcher.py
import time
import logging
from pathlib import Path
from datetime import datetime
from playwright.sync_api import sync_playwright

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger('WhatsAppWatcher')

class WhatsAppWatcher:
    def __init__(self, vault_path: str, session_path: str):
        self.vault_path = Path(vault_path)
        self.needs_action = self.vault_path / 'Needs_Action'
        self.needs_action.mkdir(exist_ok=True)

        self.session_path = Path(session_path)
        self.check_interval = 30

        # Keywords to watch for
        self.keywords = ['urgent', 'asap', 'invoice', 'payment', 'help', 'important']

        self.processed_messages = set()

    def check_for_updates(self):
        """Check WhatsApp Web for new messages."""
        messages = []

        with sync_playwright() as p:
            try:
                browser = p.chromium.launch_persistent_context(
                    str(self.session_path),
                    headless=True
                )

                page = browser.pages[0] if browser.pages else browser.new_page()
                page.goto('https://web.whatsapp.com')

                # Wait for chat list to load
                page.wait_for_selector('[data-testid="chat-list"]', timeout=30000)

                # Find chats with unread messages
                unread_chats = page.query_selector_all('[aria-label*="unread"]')

                for chat in unread_chats:
                    try:
                        text = chat.inner_text().lower()

                        # Check for keywords
                        has_keyword = any(kw in text for kw in self.keywords)

                        if has_keyword:
                            # Extract contact name
                            name_elem = chat.query_selector('[data-testid="cell-frame-title"]')
                            name = name_elem.inner_text() if name_elem else 'Unknown'

                            # Get preview text
                            preview = text[:200]

                            message_id = f"{name}_{hash(preview)}"

                            if message_id not in self.processed_messages:
                                messages.append({
                                    'contact': name,
                                    'preview': preview,
                                    'id': message_id,
                                    'keywords': [kw for kw in self.keywords if kw in text]
                                })
                                self.processed_messages.add(message_id)

                    except Exception as e:
                        logger.error(f"Error processing chat: {e}")

                browser.close()

            except Exception as e:
                logger.error(f"Error checking WhatsApp: {e}")

        return messages

    def create_action_file(self, message: dict):
        """Create action file for WhatsApp message."""
        priority = 'critical' if 'urgent' in message['keywords'] else 'high'

        content = f"""---
type: whatsapp
from: {message['contact']}
received: {datetime.now().isoformat()}
priority: {priority}
keywords: {message['keywords']}
status: pending
---

## Message Preview
{message['preview']}

## Detected Keywords
{', '.join(message['keywords'])}

## Suggested Actions
- [ ] Open WhatsApp and read full message
- [ ] Draft response
- [ ] Mark as handled
"""

        filename = f"WHATSAPP_{message['contact'].replace(' ', '_')}_{datetime.now().strftime('%Y%m%d_%H%M%S')}.md"
        filepath = self.needs_action / filename
        filepath.write_text(content)

        logger.info(f"Created: {filename}")

    def run(self):
        """Main run loop."""
        logger.info("Starting WhatsApp Watcher")
        logger.info(f"Check interval: {self.check_interval}s")
        logger.info(f"Keywords: {self.keywords}")

        while True:
            messages = self.check_for_updates()

            for msg in messages:
                self.create_action_file(msg)
                logger.info(f"New message from {msg['contact']}")

            time.sleep(self.check_interval)


if __name__ == '__main__':
    import os

    vault = os.environ.get('VAULT_PATH', '/path/to/vault')
    session = os.environ.get('WHATSAPP_SESSION_PATH', '/path/to/session')

    watcher = WhatsAppWatcher(vault, session)
    watcher.run()
```

## Example 4: Watchdog Process Manager

```python
# watchdog_manager.py
import subprocess
import time
import logging
from pathlib import Path
import json

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger('WatchdogManager')

WATCHERS = {
    'gmail_watcher': {
        'command': 'python3 gmail_watcher.py',
        'restart_delay': 5,
        'max_restarts': 10
    },
    'whatsapp_watcher': {
        'command': 'python3 whatsapp_watcher.py',
        'restart_delay': 10,
        'max_restarts': 5
    },
    'filesystem_watcher': {
        'command': 'python3 filesystem_watcher.py',
        'restart_delay': 2,
        'max_restarts': 20
    }
}

class WatchdogManager:
    def __init__(self, watchers: dict, pid_dir: str = '/tmp'):
        self.watchers = watchers
        self.pid_dir = Path(pid_dir)
        self.processes = {}
        self.restart_counts = {name: 0 for name in watchers}

    def start_watcher(self, name: str):
        """Start a watcher process."""
        config = self.watchers[name]

        logger.info(f"Starting {name}...")
        proc = subprocess.Popen(
            config['command'].split(),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )

        self.processes[name] = proc

        # Save PID
        pid_file = self.pid_dir / f"{name}.pid"
        pid_file.write_text(str(proc.pid))

        logger.info(f"{name} started with PID {proc.pid}")

    def check_watcher(self, name: str) -> bool:
        """Check if a watcher is running."""
        if name not in self.processes:
            return False

        proc = self.processes[name]
        return proc.poll() is None

    def restart_watcher(self, name: str):
        """Restart a crashed watcher."""
        config = self.watchers[name]

        if self.restart_counts[name] >= config['max_restarts']:
            logger.error(f"{name} exceeded max restarts, not restarting")
            return False

        logger.warning(f"{name} crashed, restarting in {config['restart_delay']}s...")
        time.sleep(config['restart_delay'])

        self.start_watcher(name)
        self.restart_counts[name] += 1

        return True

    def run(self):
        """Main monitoring loop."""
        logger.info("Starting Watchdog Manager")

        # Start all watchers
        for name in self.watchers:
            self.start_watcher(name)

        # Monitor loop
        while True:
            for name in self.watchers:
                if not self.check_watcher(name):
                    self.restart_watcher(name)

            # Update health status
            self.update_health_status()

            time.sleep(60)

    def update_health_status(self):
        """Update health status file."""
        status = {}
        for name in self.watchers:
            status[name] = {
                'running': self.check_watcher(name),
                'restarts': self.restart_counts[name],
                'pid': self.processes.get(name, {}).pid if name in self.processes else None
            }

        status_file = self.pid_dir / 'watcher_health.json'
        status_file.write_text(json.dumps(status, indent=2))


if __name__ == '__main__':
    manager = WatchdogManager(WATCHERS)
    manager.run()
```
