#!/usr/bin/env python3
"""
Start and manage watcher processes for the Personal AI Employee.
Usage: python start_watchers.py --vault /path/to/vault [--watchers gmail,whatsapp,filesystem]
"""

import argparse
import subprocess
import sys
import os
import time
import signal
import json
from pathlib import Path
from datetime import datetime


# Watcher configurations
WATCHER_CONFIGS = {
    'gmail': {
        'script': 'gmail_watcher.py',
        'description': 'Gmail inbox monitor',
        'env_required': ['GMAIL_CREDENTIALS', 'GMAIL_TOKEN'],
        'check_interval': 120
    },
    'whatsapp': {
        'script': 'whatsapp_watcher.py',
        'description': 'WhatsApp Web monitor',
        'env_required': ['WHATSAPP_SESSION_PATH'],
        'check_interval': 30
    },
    'filesystem': {
        'script': 'filesystem_watcher.py',
        'description': 'Drop folder monitor',
        'env_required': ['DROP_FOLDER'],
        'check_interval': 0  # Realtime
    },
    'finance': {
        'script': 'finance_watcher.py',
        'description': 'Bank transaction monitor',
        'env_required': ['BANK_API_TOKEN'],
        'check_interval': 3600
    }
}


class WatcherManager:
    def __init__(self, vault_path: str, scripts_dir: str = None):
        self.vault_path = Path(vault_path)
        self.scripts_dir = Path(scripts_dir) if scripts_dir else Path(__file__).parent
        self.processes = {}
        self.pid_file = self.vault_path / 'Logs' / 'watcher_pids.json'

        # Create logs directory
        (self.vault_path / 'Logs').mkdir(exist_ok=True)

    def check_requirements(self, watcher_name: str) -> tuple[bool, list]:
        """Check if required environment variables are set."""
        config = WATCHER_CONFIGS.get(watcher_name)
        if not config:
            return False, [f"Unknown watcher: {watcher_name}"]

        missing = []
        for env_var in config['env_required']:
            if not os.environ.get(env_var):
                missing.append(env_var)

        return len(missing) == 0, missing

    def start_watcher(self, name: str, dry_run: bool = False) -> bool:
        """Start a single watcher."""
        config = WATCHER_CONFIGS.get(name)
        if not config:
            print(f"❌ Unknown watcher: {name}")
            return False

        # Check requirements
        ready, missing = self.check_requirements(name)
        if not ready:
            print(f"❌ {name}: Missing environment variables: {', '.join(missing)}")
            return False

        script_path = self.scripts_dir / config['script']
        if not script_path.exists():
            print(f"❌ {name}: Script not found at {script_path}")
            return False

        if dry_run:
            print(f"✓ {name}: Would start {config['script']}")
            return True

        # Set vault path in environment
        env = os.environ.copy()
        env['VAULT_PATH'] = str(self.vault_path)

        # Start process
        try:
            proc = subprocess.Popen(
                [sys.executable, str(script_path)],
                env=env,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                start_new_session=True
            )

            self.processes[name] = proc
            print(f"✅ {name}: Started (PID {proc.pid})")

            return True

        except Exception as e:
            print(f"❌ {name}: Failed to start - {e}")
            return False

    def stop_watcher(self, name: str) -> bool:
        """Stop a single watcher."""
        if name not in self.processes:
            print(f"⚠️  {name}: Not running")
            return False

        proc = self.processes[name]
        try:
            proc.terminate()
            proc.wait(timeout=5)
            print(f"✅ {name}: Stopped")
            del self.processes[name]
            return True
        except subprocess.TimeoutExpired:
            proc.kill()
            print(f"⚠️  {name}: Force killed")
            del self.processes[name]
            return True
        except Exception as e:
            print(f"❌ {name}: Failed to stop - {e}")
            return False

    def start_all(self, watchers: list = None, dry_run: bool = False):
        """Start multiple watchers."""
        watchers = watchers or list(WATCHER_CONFIGS.keys())

        print(f"\n🚀 Starting watchers...")
        print(f"   Vault: {self.vault_path}")
        print(f"   Watchers: {', '.join(watchers)}\n")

        started = 0
        for name in watchers:
            if self.start_watcher(name, dry_run):
                started += 1

        print(f"\n{'Would start' if dry_run else 'Started'} {started}/{len(watchers)} watchers")

        if not dry_run:
            self.save_pids()

    def stop_all(self):
        """Stop all running watchers."""
        print("\n🛑 Stopping watchers...")

        for name in list(self.processes.keys()):
            self.stop_watcher(name)

        self.save_pids()

    def status(self):
        """Show status of all watchers."""
        print("\n📊 Watcher Status\n")
        print(f"{'Watcher':<15} {'Status':<12} {'PID':<10} {'Description'}")
        print("-" * 60)

        for name, config in WATCHER_CONFIGS.items():
            if name in self.processes:
                proc = self.processes[name]
                if proc.poll() is None:
                    status = "🟢 Running"
                    pid = str(proc.pid)
                else:
                    status = "🔴 Crashed"
                    pid = "-"
            else:
                status = "⬜ Stopped"
                pid = "-"

            print(f"{name:<15} {status:<12} {pid:<10} {config['description']}")

    def save_pids(self):
        """Save current PIDs to file."""
        pids = {name: proc.pid for name, proc in self.processes.items()}
        self.pid_file.write_text(json.dumps(pids, indent=2))

    def load_pids(self):
        """Load PIDs from file (for reconnecting)."""
        if self.pid_file.exists():
            return json.loads(self.pid_file.read_text())
        return {}


def main():
    parser = argparse.ArgumentParser(description='Manage watcher processes')
    parser.add_argument('action', choices=['start', 'stop', 'status', 'check'],
                        help='Action to perform')
    parser.add_argument('--vault', required=True, help='Path to Obsidian vault')
    parser.add_argument('--watchers', help='Comma-separated list of watchers')
    parser.add_argument('--scripts-dir', help='Directory containing watcher scripts')
    parser.add_argument('--dry-run', action='store_true', help='Show what would be done')

    args = parser.parse_args()

    manager = WatcherManager(args.vault, args.scripts_dir)

    watchers = args.watchers.split(',') if args.watchers else None

    if args.action == 'start':
        manager.start_all(watchers, args.dry_run)
    elif args.action == 'stop':
        manager.stop_all()
    elif args.action == 'status':
        manager.status()
    elif args.action == 'check':
        print("\n🔍 Checking requirements...\n")
        for name in (watchers or WATCHER_CONFIGS.keys()):
            ready, missing = manager.check_requirements(name)
            if ready:
                print(f"✅ {name}: Ready")
            else:
                print(f"❌ {name}: Missing {', '.join(missing)}")


if __name__ == '__main__':
    main()
