#!/usr/bin/env dotnet
#:package System.CommandLine@2.0.0
#:package Spectre.Console@0.49.1
#:package Spectre.Console.Cli@0.49.1
#:package CliWrap@3.6.6

#pragma warning disable IL2026
#pragma warning disable IL3050

// ============================================================================
// IMPORTANT: Wise API Limitations for Personal Accounts (PSD2 Compliance)
// ============================================================================
// As of 2024, Wise no longer supports the following operations via API for
// personal accounts due to PSD2 (Payment Services Directive 2) requirements:
//
// - Transaction history / account statements (requires SCA which is disabled)
// - FUNDING transfers via API (must be done through website/app)
// - Any operation requiring Strong Customer Authentication (SCA)
//
// You CAN still create DRAFT transfers via API - they just need to be funded
// manually through the Wise website or mobile app.
//
// For transaction history, use the archived CSV files in this directory instead.
// These are manually exported from wise.com.
//
// This tool supports: profiles, balances, recipients, quotes, draft transfers
// ============================================================================

using System.ComponentModel;
using System.Net.Http.Headers;
using System.Text;
using System.Text.Json;
using System.Text.Json.Serialization;
using CliWrap;
using CliWrap.Buffered;
using Spectre.Console.Cli;

// Setup command app
var app = new CommandApp();
app.Configure(config =>
{
    config.SetApplicationName("wise");

    config.AddCommand<ProfilesCommand>("profiles")
        .WithDescription("List all profiles (personal and business)");

    config.AddCommand<BalancesCommand>("balances")
        .WithDescription("Get balances for a profile");

    config.AddCommand<RecipientsCommand>("recipients")
        .WithDescription("List saved recipients for a profile");

    config.AddCommand<QuoteCommand>("quote")
        .WithDescription("Get a quote for a currency conversion");

    config.AddCommand<TransferCommand>("transfer")
        .WithDescription("Create a draft transfer (must be funded via website/app)");

    config.AddCommand<CreateRecipientCommand>("create-recipient")
        .WithDescription("Create a new recipient (IBAN)");
});

return app.Run(args);

// ============================================================================
// Command Handlers
// ============================================================================

class ProfilesCommand : AsyncCommand<ProfilesCommand.Settings>
{
    public class Settings : CommandSettings
    {
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var client = await WiseHelper.CreateClientAsync();
            var profiles = await client.GetProfilesAsync();

            var result = new { success = true, profiles };
            Console.WriteLine(JsonSerializer.Serialize(result, WiseHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            WiseHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

class BalancesCommand : AsyncCommand<BalancesCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Profile ID (optional, uses first profile if not specified)")]
        [CommandOption("-p|--profile-id")]
        public long? ProfileId { get; init; }

        [Description("Profile type: 'personal' or 'business'")]
        [CommandOption("-t|--type")]
        public string? ProfileType { get; init; }
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var client = await WiseHelper.CreateClientAsync();
            var profileId = settings.ProfileId ?? await client.GetProfileIdAsync(settings.ProfileType);
            var balances = await client.GetBalancesAsync(profileId);

            var result = new { success = true, profileId, balances };
            Console.WriteLine(JsonSerializer.Serialize(result, WiseHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            WiseHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

class RecipientsCommand : AsyncCommand<RecipientsCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Profile ID (optional, uses first profile if not specified)")]
        [CommandOption("-p|--profile-id")]
        public long? ProfileId { get; init; }

        [Description("Profile type: 'personal' or 'business'")]
        [CommandOption("-t|--type")]
        public string? ProfileType { get; init; }

        [Description("Filter by currency")]
        [CommandOption("-c|--currency")]
        public string? Currency { get; init; }
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var client = await WiseHelper.CreateClientAsync();
            var profileId = settings.ProfileId ?? await client.GetProfileIdAsync(settings.ProfileType);
            var recipients = await client.GetRecipientsAsync(profileId, settings.Currency);

            var result = new { success = true, profileId, recipients };
            Console.WriteLine(JsonSerializer.Serialize(result, WiseHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            WiseHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

class QuoteCommand : AsyncCommand<QuoteCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Profile ID (optional, uses first profile if not specified)")]
        [CommandOption("-p|--profile-id")]
        public long? ProfileId { get; init; }

        [Description("Profile type: 'personal' or 'business'")]
        [CommandOption("-t|--type")]
        public string? ProfileType { get; init; }

        [Description("Source currency (e.g., EUR)")]
        [CommandOption("--source-currency")]
        public required string SourceCurrency { get; init; }

        [Description("Target currency (e.g., USD)")]
        [CommandOption("--target-currency")]
        public required string TargetCurrency { get; init; }

        [Description("Amount to send (in source currency)")]
        [CommandOption("--amount")]
        public required decimal Amount { get; init; }
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var client = await WiseHelper.CreateClientAsync();
            var profileId = settings.ProfileId ?? await client.GetProfileIdAsync(settings.ProfileType);
            var quote = await client.CreateQuoteAsync(profileId, settings.SourceCurrency, settings.TargetCurrency, settings.Amount);

            var result = new { success = true, profileId, quote };
            Console.WriteLine(JsonSerializer.Serialize(result, WiseHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            WiseHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

class TransferCommand : AsyncCommand<TransferCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Profile ID (optional, uses first profile if not specified)")]
        [CommandOption("-p|--profile-id")]
        public long? ProfileId { get; init; }

        [Description("Profile type: 'personal' or 'business'")]
        [CommandOption("-t|--type")]
        public string? ProfileType { get; init; }

        [Description("Quote ID (from quote command)")]
        [CommandOption("--quote-id")]
        public required string QuoteId { get; init; }

        [Description("Recipient ID (from recipients command)")]
        [CommandOption("--recipient-id")]
        public required long RecipientId { get; init; }

        [Description("Reference/description for the transfer")]
        [CommandOption("--reference")]
        public string? Reference { get; init; }

        [Description("Confirm creation of draft transfer")]
        [CommandOption("--confirm")]
        [DefaultValue(false)]
        public bool Confirm { get; init; } = false;
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            if (!settings.Confirm)
            {
                var warning = new
                {
                    success = false,
                    error = "Transfer requires --confirm flag. This is a safety measure.",
                    hint = "First use 'quote' command to get a quote, 'recipients' to list recipients, then add --confirm to create the draft transfer.",
                    note = "Due to PSD2 requirements, the transfer will be created as a DRAFT. You must fund it via the Wise website or mobile app."
                };
                Console.WriteLine(JsonSerializer.Serialize(warning, WiseHelper.JsonOptions));
                return 1;
            }

            var client = await WiseHelper.CreateClientAsync();
            var profileId = settings.ProfileId ?? await client.GetProfileIdAsync(settings.ProfileType);
            var transfer = await client.CreateTransferAsync(settings.QuoteId, settings.RecipientId, settings.Reference);

            var result = new
            {
                success = true,
                profileId,
                transfer,
                message = "Draft transfer created. You must fund it via the Wise website or mobile app."
            };
            Console.WriteLine(JsonSerializer.Serialize(result, WiseHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            WiseHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

class CreateRecipientCommand : AsyncCommand<CreateRecipientCommand.Settings>
{
    public class Settings : CommandSettings
    {
        [Description("Profile ID (optional, uses first profile if not specified)")]
        [CommandOption("-p|--profile-id")]
        public long? ProfileId { get; init; }

        [Description("Profile type: 'personal' or 'business'")]
        [CommandOption("-t|--type")]
        public string? ProfileType { get; init; }

        [Description("Account holder name")]
        [CommandOption("--name")]
        public required string Name { get; init; }

        [Description("IBAN")]
        [CommandOption("--iban")]
        public required string Iban { get; init; }

        [Description("Currency (e.g., EUR)")]
        [CommandOption("--currency")]
        public required string Currency { get; init; }

        [Description("Legal type: PRIVATE or BUSINESS")]
        [CommandOption("--legal-type")]
        [DefaultValue("PRIVATE")]
        public string LegalType { get; init; } = "PRIVATE";
    }

    public override async Task<int> ExecuteAsync(CommandContext context, Settings settings)
    {
        try
        {
            var client = await WiseHelper.CreateClientAsync();
            var profileId = settings.ProfileId ?? await client.GetProfileIdAsync(settings.ProfileType);
            var recipient = await client.CreateRecipientAsync(
                profileId,
                settings.Name,
                settings.Iban,
                settings.Currency,
                settings.LegalType);

            var result = new { success = true, profileId, recipient };
            Console.WriteLine(JsonSerializer.Serialize(result, WiseHelper.JsonOptions));
            return 0;
        }
        catch (Exception ex)
        {
            WiseHelper.OutputError(ex.Message);
            return 1;
        }
    }
}

// ============================================================================
// Wise API Client
// ============================================================================

static class WiseHelper
{
    public static readonly JsonSerializerOptions JsonOptions = new()
    {
        WriteIndented = true,
        DefaultIgnoreCondition = JsonIgnoreCondition.WhenWritingNull,
        PropertyNamingPolicy = JsonNamingPolicy.CamelCase,
        TypeInfoResolver = new System.Text.Json.Serialization.Metadata.DefaultJsonTypeInfoResolver()
    };

    public static async Task<WiseClient> CreateClientAsync()
    {
        var apiToken = await GetKeychainPasswordAsync("wise-api", "api_token");

        if (string.IsNullOrEmpty(apiToken))
        {
            throw new Exception(
                "Wise API token not found in Keychain.\n\n" +
                "To set up, run:\n" +
                "  security add-generic-password -s \"wise-api\" -a \"api_token\" -w \"YOUR_TOKEN\" -U\n\n" +
                "Get your API token from Wise Settings > API tokens");
        }

        return new WiseClient(apiToken);
    }

    static async Task<string?> GetKeychainPasswordAsync(string service, string account)
    {
        var result = await Cli.Wrap("security")
            .WithArguments($"find-generic-password -s \"{service}\" -a \"{account}\" -w")
            .WithValidation(CommandResultValidation.None)
            .ExecuteBufferedAsync();

        if (result.ExitCode != 0)
            return null;

        return result.StandardOutput.Trim();
    }

    public static void OutputError(string message)
    {
        var error = new { success = false, error = message };
        Console.WriteLine(JsonSerializer.Serialize(error, JsonOptions));
    }
}

class WiseClient
{
    private readonly HttpClient _httpClient;
    private const string BaseUrl = "https://api.wise.com";

    public WiseClient(string apiToken)
    {
        _httpClient = new HttpClient();
        _httpClient.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Bearer", apiToken);
        _httpClient.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));
    }

    public async Task<List<JsonElement>> GetProfilesAsync()
    {
        var response = await GetAsync("/v1/profiles");
        return JsonSerializer.Deserialize<List<JsonElement>>(response, WiseHelper.JsonOptions) ?? [];
    }

    public async Task<long> GetProfileIdAsync(string? profileType)
    {
        var profiles = await GetProfilesAsync();

        foreach (var profile in profiles)
        {
            if (profile.TryGetProperty("type", out var typeElement))
            {
                var type = typeElement.GetString()?.ToLowerInvariant();
                if (profileType == null || type == profileType?.ToLowerInvariant())
                {
                    if (profile.TryGetProperty("id", out var idElement))
                    {
                        return idElement.GetInt64();
                    }
                }
            }
        }

        throw new Exception($"No profile found{(profileType != null ? $" of type '{profileType}'" : "")}");
    }

    public async Task<List<JsonElement>> GetBalancesAsync(long profileId)
    {
        var response = await GetAsync($"/v4/profiles/{profileId}/balances?types=STANDARD,SAVINGS");
        return JsonSerializer.Deserialize<List<JsonElement>>(response, WiseHelper.JsonOptions) ?? [];
    }

    public async Task<List<JsonElement>> GetRecipientsAsync(long profileId, string? currency)
    {
        var url = $"/v1/accounts?profile={profileId}";
        if (currency != null)
        {
            url += $"&currency={currency.ToUpperInvariant()}";
        }

        var response = await GetAsync(url);
        return JsonSerializer.Deserialize<List<JsonElement>>(response, WiseHelper.JsonOptions) ?? [];
    }

    public async Task<JsonElement> CreateQuoteAsync(long profileId, string sourceCurrency, string targetCurrency, decimal amount)
    {
        // v3 API - profile ID goes in URL path, not body
        var payload = new
        {
            sourceCurrency = sourceCurrency.ToUpperInvariant(),
            targetCurrency = targetCurrency.ToUpperInvariant(),
            sourceAmount = amount,
            payOut = "BALANCE"
        };

        var response = await PostAsync($"/v3/profiles/{profileId}/quotes", JsonSerializer.Serialize(payload, WiseHelper.JsonOptions));
        return JsonSerializer.Deserialize<JsonElement>(response, WiseHelper.JsonOptions);
    }

    public async Task<JsonElement> CreateTransferAsync(string quoteId, long recipientId, string? reference)
    {
        var payload = new
        {
            targetAccount = recipientId,
            quoteUuid = quoteId,
            customerTransactionId = Guid.NewGuid().ToString(),
            details = new
            {
                reference = reference ?? "Transfer via API"
            }
        };

        var response = await PostAsync("/v1/transfers", JsonSerializer.Serialize(payload, WiseHelper.JsonOptions));
        return JsonSerializer.Deserialize<JsonElement>(response, WiseHelper.JsonOptions);
    }

    public async Task<JsonElement> CreateRecipientAsync(long profileId, string name, string iban, string currency, string legalType)
    {
        var payload = new
        {
            profile = profileId,
            accountHolderName = name,
            currency = currency.ToUpperInvariant(),
            type = "iban",
            details = new
            {
                legalType = legalType.ToUpperInvariant(),
                IBAN = iban.Replace(" ", "").ToUpperInvariant()
            }
        };

        var response = await PostAsync("/v1/accounts", JsonSerializer.Serialize(payload, WiseHelper.JsonOptions));
        return JsonSerializer.Deserialize<JsonElement>(response, WiseHelper.JsonOptions);
    }

    private async Task<string> GetAsync(string path)
    {
        var response = await _httpClient.GetAsync(BaseUrl + path);
        var content = await response.Content.ReadAsStringAsync();

        if (!response.IsSuccessStatusCode)
        {
            throw new Exception($"Wise API error ({response.StatusCode}): {content}");
        }

        return content;
    }

    private async Task<string> PostAsync(string path, string body)
    {
        var response = await _httpClient.PostAsync(
            BaseUrl + path,
            new StringContent(body, Encoding.UTF8, "application/json"));

        var content = await response.Content.ReadAsStringAsync();

        if (!response.IsSuccessStatusCode)
        {
            throw new Exception($"Wise API error ({response.StatusCode}): {content}");
        }

        return content;
    }
}
